# JerryScript types

## jerry_init_flag_t

Enum that contains the following elements:

 - JERRY_INIT_EMPTY - empty flag set
 - JERRY_INIT_SHOW_OPCODES - dump byte-code to log after parse
 - JERRY_INIT_SHOW_REGEXP_OPCODES - dump regexp byte-code to log after compilation
 - JERRY_INIT_MEM_STATS - dump memory statistics
 - JERRY_INIT_MEM_STATS_SEPARATE - dump memory statistics and reset peak values after parse
 - JERRY_INIT_DEBUGGER - enable all features required by debugging

## jerry_error_t

Possible types of an error:

 - JERRY_ERROR_COMMON - common error
 - JERRY_ERROR_EVAL - eval error
 - JERRY_ERROR_RANGE - range error
 - JERRY_ERROR_REFERENCE - reference error
 - JERRY_ERROR_SYNTAX - syntax error
 - JERRY_ERROR_TYPE - type error
 - JERRY_ERROR_URI - URI error

## jerry_feature_t

Possible compile time enabled feature types:

 - JERRY_FEATURE_CPOINTER_32_BIT - 32 bit compressed pointers
 - JERRY_FEATURE_ERROR_MESSAGES - error messages
 - JERRY_FEATURE_JS_PARSER - js-parser
 - JERRY_FEATURE_MEM_STATS - memory statistics
 - JERRY_FEATURE_PARSER_DUMP - parser byte-code dumps
 - JERRY_FEATURE_REGEXP_DUMP - regexp byte-code dumps
 - JERRY_FEATURE_SNAPSHOT_SAVE - saving snapshot files
 - JERRY_FEATURE_SNAPSHOT_EXEC - executing snapshot files
 - JERRY_FEATURE_DEBUGGER - debugging
 - JERRY_FEATURE_VM_EXEC_STOP - stopping ECMAScript execution

## jerry_char_t

**Summary**

Jerry's char value

**Prototype**

```c
typedef uint8_t jerry_char_t;
```

## jerry_char_ptr_t

**Summary**

Pointer to an array of character values

**Prototype**

```c
typedef jerry_char_t *jerry_char_ptr_t;
```

## jerry_size_t

**Summary**

Jerry's size

**Prototype**

```c
typedef uint32_t jerry_size_t;
```

## jerry_length_t

**Summary**

Jerry's length

**Prototype**

```c
typedef uint32_t jerry_length_t;
```

## jerry_value_t

**Summary**

JerryScript value can be a boolean, number, null, object, string or undefined. The value has an error flag,
that indicates whether is an error or not. Every type has an error flag not only objects. The error flag should
be cleared before the value is passed as an argument, otherwise it can lead to a type error. The error objects
created by API functions has the error flag set.

**Prototype**

```c
typedef uint32_t jerry_value_t;
```

## jerry_context_data_manager_t

**Summary**

Structure that defines how a context data item will be initialized and deinitialized. JerryScript zeroes out the memory
for the item by default, and if the `init_cb` field is not NULL, it will be called with the pointer to the memory  as
an additional custom initializer.

**Prototype**

```c
typedef struct
{
  void (*init_cb) (void *); /**< callback responsible for initializing a context item, or NULL */
  void (*deinit_cb) (void *); /**< callback responsible for deinitializing a context item */
  size_t bytes_needed; /**< number of bytes to allocate for this manager */
} jerry_context_data_manager_t;
```

## jerry_property_descriptor_t

**Summary**

Description of ECMA property descriptor

**Prototype**

```c
typedef struct
{
  /** Is [[Value]] defined? */
  bool is_value_defined;

  /** Is [[Get]] defined? */
  bool is_get_defined;

  /** Is [[Set]] defined? */
  bool is_set_defined;

  /** Is [[Writable]] defined? */
  bool is_writable_defined;

  /** [[Writable]] */
  bool is_writable;

  /** Is [[Enumerable]] defined? */
  bool is_enumerable_defined;

  /** [[Enumerable]] */
  bool is_enumerable;

  /** Is [[Configurable]] defined? */
  bool is_configurable_defined;

  /** [[Configurable]] */
  bool is_configurable;

  /** [[Value]] */
  jerry_value_t value;

  /** [[Get]] */
  jerry_value_t getter;

  /** [[Set]] */
  jerry_value_t setter;
} jerry_property_descriptor_t;
```

## jerry_external_handler_t

**Summary**

Type of an external function handler

**Prototype**

```c
typedef jerry_value_t (*jerry_external_handler_t) (const jerry_value_t function_obj,
                                                   const jerry_value_t this_val,
                                                   const jerry_value_t args_p[],
                                                   const jerry_length_t args_count);
```

## jerry_object_free_callback_t

**Summary**

**Deprecated: Please use jerry_object_native_free_callback_t instead.**

Native free callback of an object.

**Prototype**

```c
typedef void (*jerry_object_free_callback_t) (const uintptr_t native_p);
```

## jerry_object_native_free_callback_t

**Summary**

Native free callback of an object. It is used in jerry_object_native_info_t.

**Prototype**

```c
typedef void (*jerry_object_native_free_callback_t) (void *native_p);
```

## jerry_object_native_info_t

**Summary**

The type infomation of the native pointer.
It includes the free callback that will be called when associated JavaScript object is garbage collected. It can be left NULL in case it is not needed.

Typically, one would create a `static const jerry_object_native_info_t` for
each distinct C type for which a pointer is used with
`jerry_set_object_native_pointer ()` and `jerry_get_object_native_pointer ()`.
This way, each `const jerry_object_native_info_t *` pointer address value itself
uniquely identifies the C type of the native pointer.

See [jerry_get_object_native_pointer](#jerry_get_object_native_pointer)
for a best-practice code example.

**Prototype**

```c
typedef struct
{
  jerry_object_native_free_callback_t free_cb;
} jerry_object_native_info_t;
```

**See also**

- [jerry_set_object_native_pointer](#jerry_set_object_native_pointer)
- [jerry_get_object_native_pointer](#jerry_get_object_native_pointer)

## jerry_object_property_foreach_t

**Summary**

Function type applied for each data property of an object

**Prototype**

```c
typedef bool (*jerry_object_property_foreach_t) (const jerry_value_t property_name,
                                                 const jerry_value_t property_value,
                                                 void *user_data_p);
```

## jerry_vm_exec_stop_callback_t

**Summary**

Callback which tells whether the ECMAScript execution should be stopped.
If it returns with undefined value the ECMAScript execution continues.
Otherwise the result is thrown by the engine (if the error flag is not
set for the returned value the engine automatically sets it). The
callback function might be called again even if it threw an error.
In this case the function must throw the same error again.

**Prototype**

```c
typedef jerry_value_t (*jerry_vm_exec_stop_callback_t) (void *user_p);
```

**See also**

- [jerry_set_vm_exec_stop_callback](#jerry_set_vm_exec_stop_callback)


# General engine functions

## jerry_init

**Summary**

Initializes the JerryScript engine, making it possible to run JavaScript code and perform operations
on JavaScript values.

**Prototype**

```c
void
jerry_init (jerry_init_flag_t flags)
```

`flags` - combination of various engine configuration flags:

- `JERRY_INIT_EMPTY` - no flags, just initialize in default configuration.
- `JERRY_INIT_SHOW_OPCODES` - print compiled byte-code.
- `JERRY_INIT_SHOW_REGEXP_OPCODES` - print compiled regexp byte-code.
- `JERRY_INIT_MEM_STATS` - dump memory statistics.
- `JERRY_INIT_MEM_STATS_SEPARATE` - dump memory statistics and reset peak values after parse.
- `JERRY_INIT_DEBUGGER` - enable all features required by debugging.

**Example**

```c
{
  jerry_init (JERRY_INIT_SHOW_OPCODES | JERRY_INIT_SHOW_REGEXP_OPCODES);

  // ...

  jerry_cleanup ();
}
```

**See also**

- [jerry_cleanup](#jerry_cleanup)


## jerry_cleanup

**Summary**

Finish JavaScript engine execution, freeing memory and JavaScript values.

*Note*: JavaScript values, received from engine, will be inaccessible after the cleanup.

**Prototype**

```c
void
jerry_cleanup (void);
```

**See also**

- [jerry_init](#jerry_init)


## jerry_get_context_data

**Summary**

Retrieve a pointer to the item stored within the current context by the given manager.

*Note*: Since internally the pointer to a manager's context data item is linked to the next such pointer in a linked
        list, it is inadvisable to invoke too many different managers, because doing so will increase the time it takes
        to retrieve a manager's context data item, degrading performance. For example, try to keep the number of
        managers below five.

**Prototype**

```c
void *
jerry_get_context_data (const jerry_context_data_manager *manager_p);
```

- `manager_p`: the manager of this context data item.
- return value: the item created by `manager_p` when `jerry_get_context_data ()` was first called, or a new item created
  by `manager_p`, which will be stored for future identical calls to `jerry_get_context_data ()`, and which will be
  deinitialized using the `deinit_cb` callback provided by `manager_p` when the context will be destroyed.

**Example**

```c
typedef struct
{
  int my_data1;
  double my_data2;
  char *my_data3;
} my_context_data_t;

/* Define how context items will be initialized. */
static void
my_context_data_new (void *user_data_p)
{
  my_context_data_t *my_data_p = (my_context_data_t *) user_data_p;

  /*
   * Initialize my_data_p. JerryScript will store it on the current context and return it whenever
   * jerry_get_context_data () is called with a pointer to my_manager as defined below.
   */
}

/* Define how context items will be deinitialized */
static void
my_context_data_free (void *user_data_p)
{
  my_context_data_t *my_data_p = ((my_context_data_t *) user_data_p);

  /* Perform any necessary cleanup on my_data. JerryScript will free the pointer after this function completes. */
}

/* Wrap the creation and destruction functions into a manager */
static const jerry_context_data_manager_t my_manager =
{
  .init_cb = my_context_data_new,
  .deinit_cb = my_context_data_free,
  .bytes_needed = sizeof (my_context_data_t)
};

/*
 * Then, in some function in your code, you can retrieve an item of type my_context_data_t from the currently active
 * context such that JerryScript will create and store such an item if one was not previously created
 */
void someplace_in_the_code (void)
{
  my_context_data_t *my_data = (my_context_data_t *) jerry_get_context_data (&my_manager);
  /* Perform useful things using the data found in my_data */
}
```


## jerry_register_magic_strings

**Summary**

Registers an external magic string array.

*Note*: The strings in the array must be sorted by size at first, then lexicographically.

**Prototype**

```c
void
jerry_register_magic_strings  (const jerry_char_ptr_t *ex_str_items_p,
                               uint32_t count,
                               const jerry_length_t *str_lengths_p);
```

- `ex_str_items_p` - character arrays, representing external magic strings' contents
- `count` - number of the strings
- `str_lengths_p` - lengths of the strings

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  // must be static, because 'jerry_register_magic_strings' does not copy
  // the items must be sorted by size at first, then lexicographically
  static const jerry_char_ptr_t magic_string_items[] = {
                                                         (const jerry_char_ptr_t) "magicstring1",
                                                         (const jerry_char_ptr_t) "magicstring2",
                                                         (const jerry_char_ptr_t) "magicstring3"
                                                       };
  uint32_t num_magic_string_items = (uint32_t) (sizeof (magic_string_items) / sizeof (jerry_char_ptr_t));

  // must be static, because 'jerry_register_magic_strings' does not copy
  static const jerry_length_t magic_string_lengths[] = {
                                                         (jerry_length_t)strlen (magic_string_items[0]),
                                                         (jerry_length_t)strlen (magic_string_items[1]),
                                                         (jerry_length_t)strlen (magic_string_items[2])
                                                       };
  jerry_register_magic_strings (magic_string_items, num_magic_string_items, magic_string_lengths);
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_parse_and_save_literals](#jerry_parse_and_save_literals)


## jerry_get_memory_limits

**Summary**

Gets configured memory limits of JerryScript.

**Prototype**

```c
void
jerry_get_memory_limits (size_t *out_data_bss_brk_limit_p,
                         size_t *out_stack_limit_p);
```

- `out_data_bss_brk_limit_p` - out parameter, that gives the maximum size of data + bss + brk sections.
- `out_stack_limit_p` - out parameter, that gives the maximum size of the stack.

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  size_t stack_limit;
  size_t data_bss_brk_limit;
  jerry_get_memory_limits (&stack_limit, &data_bss_brk_limit);
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)


## jerry_gc

**Summary**

Performs garbage collection.

**Prototype**

```c
void
jerry_gc (void);
```

**Example**

```c
jerry_gc ();
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)

# Parser and executor functions

Functions to parse and run JavaScript source code.

## jerry_run_simple

**Summary**

The simplest way to run JavaScript.

**Prototype**

```c
bool
jerry_run_simple (const jerry_char_t *script_source_p,
                  size_t script_source_size,
                  jerry_init_flag_t flags);
```

- `script_source_p` - source code, it must be a valid utf8 string.
- `script_source_size` - size of source code buffer, in bytes.
- `jerry_init_flag_t` - combination of various engine configuration flags
- return value
  - true, if run was successful
  - false, otherwise

**Example**

```c
{
  const jerry_char_t *script = "print ('Hello, World!');";

  jerry_run_simple (script, strlen ((const char *) script), JERRY_INIT_EMPTY);
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_parse](#jerry_parse)
- [jerry_run](#jerry_run)


## jerry_parse

**Summary**

Parse script and construct an EcmaScript function. The
lexical environment is set to the global lexical environment.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_parse (const jerry_char_t *source_p,
             size_t source_size,
             bool is_strict);
```

- `source_p` - string, containing source code to parse. It must be a valid utf8 string.
- `source_size` - size of the string, in bytes.
- `is_strict` - defines strict mode.
- return value
  - function object value, if script was parsed successfully,
  - thrown error, otherwise

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "print ('Hello, World!');";
  size_t script_size = strlen ((const char *) script);

  jerry_value_t parsed_code = jerry_parse (script, script_size, false);
  jerry_release_value (parsed_code);

  jerry_cleanup ();
}
```

**See also**

- [jerry_run](#jerry_run)

## jerry_parse_named_resource

**Summary**

Parse script and construct an ECMAScript function. The lexical
environment is set to the global lexical environment. The name
(usually a file name) is also passed to this function which is
used by the debugger to find the source code.

*Note*: The returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_parse_named_resource (const jerry_char_t *name_p, /**< name (usually a file name) */
                            size_t name_length, /**< length of name */
                            const jerry_char_t *source_p, /**< script source */
                            size_t source_size, /**< script source size */
                            bool is_strict) /**< strict mode */
{
```

- `name_p` - name, usually a file name
- `name_length` - size of the file name, in bytes
- `source_p` - string, containing source code to parse. It must be a valid UTF8 string
- `source_size` - size of the string, in bytes
- `is_strict` - defines strict mode
- return value
  - function object value, if script was parsed successfully,
  - thrown error, otherwise

This function is identical to [jerry_parse](#jerry_parse), except that an additional filename parameter has been added.

## jerry_run

**Summary**

Run an EcmaScript function created by `jerry_parse`.

*Note*: The code should be previously parsed with `jerry_parse`.

**Prototype**

```c
jerry_value_t
jerry_run (const jerry_value_t func_val);
```

- `func_val` - function to run
- return value
  - result of bytecode, if run was successful
  - thrown error, otherwise

**Example**

```c
{
  const jerry_char_t script[] = "print ('Hello, World!');";
  size_t script_size = strlen ((const char *) script);

  /* Initialize engine */
  jerry_init (JERRY_INIT_EMPTY);

  /* Setup Global scope code */
  jerry_value_t parsed_code = jerry_parse (script, script_size, false);

  if (!jerry_value_has_error_flag (parsed_code))
  {
    /* Execute the parsed source code in the Global scope */
    jerry_value_t ret_value = jerry_run (parsed_code);

    /* Returned value must be freed */
    jerry_release_value (ret_value);
  }

  /* Parsed source code must be freed */
  jerry_release_value (parsed_code);

  /* Cleanup engine */
  jerry_cleanup ();
}
```

**See also**

- [jerry_parse](#jerry_parse)


## jerry_eval

**Summary**

Perform JavaScript `eval`.

**Prototype**

```c
jerry_value_t
jerry_eval (const jerry_char_t *source_p,
            size_t source_size,
            bool is_strict);
```

- `source_p` - source code to evaluate, it must be a valid utf8 string.
- `source_size` - length of the source code
- `is_strict` - perform `eval` as it is called from "strict mode" code.
- return value - result of eval, may be error value.

**Example**

```c
{
  jerry_value_t ret_val = jerry_eval (str_to_eval,
                                      strlen (str_to_eval),
                                      false);
}
```

**See also**

- [jerry_create_external_function](#jerry_create_external_function)
- [jerry_external_handler_t](#jerry_external_handler_t)

## jerry_run_all_enqueued_jobs

**Summary**

Run enqueued Promise jobs until the first thrown error or until all get executed.

**Prototype**

```c
jerry_value_t
jerry_run_all_enqueued_jobs (void)
```

- return value - result of last executed job, may be error value.

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "new Promise(function(f,r) { f('Hello, World!'); }).then(function(x) { print(x); });";
  size_t script_size = strlen ((const char *) script);

  jerry_value_t parsed_code = jerry_parse (script, script_size, false);
  jerry_value_t script_value = jerry_run (parsed_code);
  jerry_value_t job_value = jerry_run_all_enqueued_jobs ();

  jerry_release_value (job_value);
  jerry_release_value (script_value);
  jerry_release_value (parsed_code);

  jerry_cleanup ();
}
```


# Get the global context

## jerry_get_global_object

**Summary**

Get the Global object.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_global_object (void);
```

- return value - api value of global object

**Example**

```c
{
  jerry_value_t glob_obj_val = jerry_get_global_object ();

  ... // Do something with global object, ex: add properties

  jerry_release_value (glob_obj_val);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_define_own_property](#jerry_define_own_property)

# Checker functions

Functions to check the type of an API value ([jerry_value_t](#jerry_value_t)).

## jerry_value_is_array

**Summary**

Returns whether the given `jerry_value_t` is an array.

**Prototype**

```c
bool
jerry_value_is_array (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an array
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_array (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_boolean

**Summary**

Returns whether the given `jerry_value_t` is a boolean value.

**Prototype**

```c
bool
jerry_value_is_boolean (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a boolean value
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_boolean (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_constructor

**Summary**

Returns whether the given `jerry_value_t` is a constructor function.

**Prototype**

```c
bool
jerry_value_is_constructor (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a constructor
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_constructor (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_function

**Summary**

Returns whether the given `jerry_value_t` is a function.

**Prototype**

```c
bool
jerry_value_is_function (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a function
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_function (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_number

**Summary**

Returns whether the given `jerry_value_t` is a number.

**Prototype**

```c
bool
jerry_value_is_function (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a number
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_number (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_null

**Summary**

Returns whether the given `jerry_value_t` is a null value.

**Prototype**

```c
bool
jerry_value_is_null (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a null
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_null (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_object

**Summary**

Returns whether the given `jerry_value_t` is an object value.

**Prototype**

```c
bool
jerry_value_is_object (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an object
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_object (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_promise

**Summary**

Returns whether the given `jerry_value_t` is a promise value.

*Note*: This API depends on the ES2015-subset profile.

**Prototype**

```c
bool
jerry_value_is_promise (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a promise
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_promise (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_string

**Summary**

Returns whether the given `jerry_value_t` is a string value.

**Prototype**

```c
bool
jerry_value_is_string (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a string
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_string (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_undefined

**Summary**

Returns whether the given `jerry_value_t` is an undefined value.

**Prototype**

```c
bool
jerry_value_is_undefined (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an undefined value
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_undefined (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)

## jerry_is_feature_enabled

**Summary**

Returns whether the specified compile time feature is enabled.

**Prototype**

```c
bool
jerry_is_feature_enabled (const jerry_feature_t feature);
```

- `feature` - jerry feature
- return value
  - true, if the given `jerry_feature_t` is enabled
  - false, otherwise

**Example**

```c
{
  ...
  jerry_feature_t feature = JERRY_FEATURE_SNAPSHOT_SAVE;

  if (jerry_is_feature_enabled (feature))
  {
    ...
  }

}
```

# Error flag manipulation functions

## jerry_value_has_error_flag

**Summary**

Returns whether the given `jerry_value_t` has the error flag set.

**Prototype**

```c
bool
jerry_value_has_error_flag (const jerry_value_t value);
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` has the error flag set
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_has_error_flag (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)


## jerry_value_clear_error_flag

**Summary**

Clear the error flag.

**Prototype**

```c
void
jerry_value_clear_error_flag (jerry_value_t *value_p);
```

- `value_p` - pointer to an api value

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_clear_error_flag (&value);

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)


## jerry_value_set_error_flag

**Summary**

Set the error flag.

**Prototype**

```c
void
jerry_value_set_error_flag (jerry_value_t *value_p);
```

- `value_p` - pointer to an api value

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_set_error_flag (&value);

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)


# Getter functions of 'jerry_value_t'

Get raw data from API values.

## jerry_get_boolean_value

**Summary**

Gets the raw bool value from a `jerry_value_t`.

**Prototype**

```c
bool
jerry_get_boolean_value (const jerry_value_t value);
```

- `value` - api value
- return value - boolean value represented by the argument.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_boolean (value))
  {
    bool raw_value = jerry_get_boolean_value (value);

    ... // usage of raw value

  }

  jerry_release_value (value);
}

```

**See also**

- [jerry_value_is_boolean](#jerry_value_is_boolean)
- [jerry_release_value](#jerry_release_value)


## jerry_get_number_value

**Summary**

Gets the number value of the given `jerry_value_t` parameter as a raw double.

**Prototype**

```c
double
jerry_get_number_value (const jerry_value_t value);
```

- `value` - api value
- return value - the number value of the given `jerry_value_t` parameter as a raw double.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_number (value))
  {
    double raw_value = jerry_get_number_value (value);

    ... // usage of raw value

  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_is_number](#jerry_value_is_number)
- [jerry_release_value](#jerry_release_value)


# Functions for string values

## jerry_get_string_size

**Summary**

Get the size of a string. Returns zero, if the value parameter is not a string.

**Prototype**

```c
jerry_size_t
jerry_get_string_size (const jerry_value_t value);
```
- `value` - api value
- return value - number of bytes in the buffer needed to represent the string.

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string = jerry_create_string (char_array);

  jerry_size_t string_size = jerry_get_string_size (string);

  ... // usage of string_size

  jerry_release_value (string);
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_length](#jerry_get_string_length)
- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)


## jerry_get_utf8_string_size

**Summary**

Get the size of an utf8-encoded string. Returns zero, if the value parameter is not a string.

*Note*: The difference from [jerry_get_string_size](#jerry_get_string_size) is that it returns with utf-8 string size
instead of the cesu-8 string size.

**Prototype**

```c
jerry_size_t
jerry_get_utf8_string_size (const jerry_value_t value);
```
- `value` - api value
- return value - number of bytes in the buffer needed to represent the utf8-encoded string.

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string = jerry_create_string (char_array);

  jerry_size_t string_size = jerry_get_utf8_string_size (string);

  ... // usage of string_size

  jerry_release_value (string);
}
```

**See also**

- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)
- [jerry_get_utf8_string_length](#jerry_get_utf8_string_length)
- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)


## jerry_get_string_length

**Summary**

Get the length of a string. Returns zero, if the value parameter is not a string.

**Prototype**

```c
jerry_length_t
jerry_get_string_length (const jerry_value_t value);
```

- `value` - api value
- return value - number of characters in the string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string = jerry_create_string (char_array);

  jerry_length_t string_length = jerry_get_string_length (string);

  ... // usage of string_length

  jerry_release_value (string);
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_size](#jerry_get_string_size)
- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)


## jerry_get_utf8_string_length

**Summary**

Get the length of an UTF-8 encoded string. Returns zero, if the value parameter is not a string.

*Note*: The difference from [jerry_get_string_length](#jerry_get_string_length) is that it
returns with utf-8 string length instead of the cesu-8 string length.

**Prototype**

```c
jerry_length_t
jerry_get_utf8_string_length (const jerry_value_t value);
```

- `value` - input string value
- return value - number of characters in the string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string = jerry_create_string_from_utf8 (char_array);

  jerry_length_t string_length = jerry_get_utf8_string_length (string);

  ... // usage of string_length

  jerry_release_value (string);
}
```

**See also**

- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)
- [jerry_get_utf8_string_size](#jerry_get_utf8_string_size)
- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)


## jerry_string_to_char_buffer

**Summary**

Copy the characters of a string into a specified cesu-8 buffer.
The '\0' character could occur in the character buffer. Returns 0,
if the value parameter is not a string or the buffer is not large
enough for the whole string.

*Note*: Does not put '\0' to the end of string, the return value identifies
the number of valid bytes in the output buffer.

**Prototype**

```c
jerry_size_t
jerry_string_to_char_buffer (const jerry_value_t value,
                             jerry_char_t *buffer_p,
                             jerry_size_t buffer_size);
```

- `value` - input string value
- `buffer_p` - pointer to output buffer
- `buffer_size` - size of the buffer
- return value - number of bytes, actually copied to the buffer

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_size_t req_sz = jerry_get_string_size (value);
  jerry_char_t str_buf_p[req_sz];

  jerry_string_to_char_buffer (value, str_buf_p, req_sz);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_size](#jerry_get_string_size)
- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)


## jerry_string_to_utf8_char_buffer

**Summary**

Copy the characters of a string into a specified utf-8 buffer.
The '\0' character could occur in character buffer. Returns 0,
if the value parameter is not a string or the buffer is not
large enough for the whole string.

*Note*: Does not put '\0' to the end of string, the return value identifies
the number of valid bytes in the output buffer.

**Prototype**

```c
jerry_size_t
jerry_string_to_utf8_char_buffer (const jerry_value_t value,
                                  jerry_char_t *buffer_p,
                                  jerry_size_t buffer_size);
```

- `value` - input string value
- `buffer_p` - pointer to output buffer
- `buffer_size` - size of the buffer
- return value - number of bytes, actually copied to the buffer

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_size_t req_sz = jerry_get_utf8_string_size (value);
  jerry_char_t str_buf_p[req_sz];

  jerry_string_to_utf8_char_buffer (value, str_buf_p, req_sz);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)
- [jerry_get_utf8_string_size](#jerry_get_utf8_string_size)
- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)


## jerry_substring_to_char_buffer

**Summary**

Copy the characters of a cesu-8 encoded substring into a specified buffer.
The '\0' character could occur in character buffer. Returns 0, if the value
parameter is not a string. It will extract the substring between the
specified start position and the end position (or the end of the string,
whichever comes first).

*Note*: Does not put '\0' to the end of string, the return value identifies
the number of valid bytes in the output buffer.

**Prototype**

```c
jerry_size_t
jerry_substring_to_char_buffer (const jerry_value_t value,
                                jerry_length_t start_pos,
                                jerry_length_t end_pos,
                                jerry_char_t *buffer_p,
                                jerry_size_t buffer_size);
```

- `value` - input string value
- `start_pos` - position of the first character
- `end_pos` - position of the last character
- `buffer_p` - pointer to output buffer
- `buffer_size` - size of the buffer
- return value - number of bytes, actually copied to the buffer

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_size_t req_sz = jerry_get_string_size (value);
  jerry_char_t str_buf_p[req_sz];
  jerry_length_t start_pos = 0;
  jerry_length_t end_pos = jerry_get_string_length (value);

  jerry_substring_to_char_buffer (value, start_pos, end_pos, str_buf_p, req_sz);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_size](#jerry_get_string_size)
- [jerry_get_string_length](#jerry_get_string_length)
- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)


## jerry_substring_to_utf8_char_buffer

**Summary**

Copy the characters of an utf-8 encoded substring into a specified buffer.
The '\0' character could occur in character buffer. Returns 0, if the value
parameter is not a string. It will extract the substring between the specified
start position and the end position (or the end of the string, whichever
comes first).

*Note*: Does not put '\0' to the end of string, the return value identifies
the number of valid bytes in the output buffer.

**Prototype**

```c
jerry_size_t
jerry_substring_to_utf8_char_buffer (const jerry_value_t value,
                                     jerry_length_t start_pos,
                                     jerry_length_t end_pos,
                                     jerry_char_t *buffer_p,
                                     jerry_size_t buffer_size);
```

- `value` - input string value
- `start_pos` - position of the first character
- `end_pos` - position of the last character
- `buffer_p` - pointer to output buffer
- `buffer_size` - size of the buffer
- return value - number of bytes, actually copied to the buffer

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_size_t req_sz = jerry_get_utf8_string_size (value);
  jerry_char_t str_buf_p[req_sz];
  jerry_length_t start_pos = 0;
  jerry_length_t end_pos = jerry_get_utf8_string_length (value);

  jerry_substring_to_utf8_char_buffer (value, start_pos, end_pos, str_buf_p, req_sz);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)
- [jerry_get_utf8_string_size](#jerry_get_utf8_string_size)
- [jerry_get_utf8_string_length](#jerry_get_utf8_string_length)
- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)


# Functions for array object values

## jerry_get_array_length

**Summary**

Get length of an array object. Returns zero, if the given parameter is not an array object.

**Prototype**

```c
uint32_t
jerry_get_array_length (const jerry_value_t value);
```

- `value` - input array value
- return value - length of the given array

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  uint32_t len = jerry_get_array_length (value);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_array](#jerry_create_array)


# Converters of 'jerry_value_t'

Functions for converting API values to another value type.

## jerry_value_to_boolean

**Summary**

Call ToBoolean operation on the api value.

**Prototype**

```c
bool
jerry_value_to_boolean (const jerry_value_t value);
```

- `value` - api value
- return value
  - true, if the logical value is true
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  bool b = jerry_value_to_boolean (value);

  jerry_release_value (value);
}

```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_number

**Summary**

Call ToNumber operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_number (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted number value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t number_value = jerry_value_to_number (value);

  jerry_release_value (number_value);
  jerry_release_value (value);
}

```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_object

**Summary**

Call ToObject operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_object (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted object value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t object_value = jerry_value_to_object (value);

  jerry_release_value (object_value);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_primitive

**Summary**

Call ToPrimitive operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_primitive (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted primitive value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t prim_value = jerry_value_to_primitive (value);

  jerry_release_value (prim_value);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)

## jerry_value_to_string

**Summary**

Call the ToString ecma builtin operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_string (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted string value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t string_value = jerry_value_to_string (value);

  jerry_release_value (string_value);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)


# Functions for promise objects

These APIs all depends on the ES2015-subset profile.

## jerry_resolve_or_reject_promise

**Summary**

Resolve or reject the promise with an argument.

**Prototype**

```c
jerry_value_t
jerry_resolve_or_reject_promise (jerry_value_t promise,
                                 jerry_value_t argument,
                                 bool is_resolve)
```

- `promise` - the promise value
- `argument` - the argument for resolve or reject
- `is_resolve` - whether the promise should be resolved or rejected
- return value
  - undefined jerry value - resolve or reject successed
  - jerry value with error flag - otherwise

**Example**

```c
{
  jerry_value_t promise = ... // acquire/create a promise object.

  ...

  bool is_resolve = ... // whether the promise should be resolved or rejected
  jerry_value_t argument = ... // prepare the argumnent for the resolve or reject.

  jerry_value_t is_ok = jerry_resolve_or_reject_promise (promise,
                                                         argument,
                                                         is_resolve);

  if (jerry_value_has_error_flag (is_ok))
  {
    // handle the error.
  }

  jerry_release_value (is_ok);
  jerry_release_value (argument);
  jerry_release_value (promise);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_value_has_error_flag](#jerry_value_has_error_flag)


# Acquire and release API values

## jerry_acquire_value

**Summary**

Acquires the specified Jerry API value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_acquire_value (jerry_value_t value);
```

- `value` - api value
- return value - acquired value that may be used outside of the engine

**Example**

```c
{
  jerry_value_t object_value = jerry_create_object ();

  jerry_value_t acquired_object = jerry_acquire_value (object_value);

  jerry_release_value (object_value);

  // acquired_object refers to the created object and makes it
  // available after the release of 'object_value'

  jerry_release_value (acquired_object);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_value_t](#jerry_value_t)


## jerry_release_value

**Summary**

Release specified Jerry API value.

**Prototype**

```c
void
jerry_release_value (jerry_value_t value);
```

- `value` - api value

**Example**

```c
{
  jerry_value_t object_value = jerry_create_object ();

  ...

  jerry_release_value (object_value);
}
```


# Create API values

Function for creating [API values](#jerry_value_t).

*Note*: Every created API value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

## jerry_create_array

**Summary**

Create an array object value.

**Prototype**

```c
jerry_value_t
jerry_create_array (uint32_t size);
```

 - `size` - size of array;
 - return value - value of the constructed array object

 **Example**

```c
{
    jerry_value_t array = jerry_create_array (10);

    ...

    jerry_release_value (array);
}
```

**See also**

- [jerry_set_property_by_index](#jerry_set_property_by_index)
- [jerry_get_property_by_index](#jerry_get_property_by_index)


## jerry_create_boolean

**Summary**

Create a jerry_value_t representing a boolean value from the given boolean parameter.

**Prototype**

```c
jerry_value_t
jerry_create_boolean (bool value);
```

- `value` - raw boolean value.
- return value - a `jerry_value_t` created from the given boolean argument.

**Example**

```c
{
  jerry_value_t boolean_value = jerry_create_boolean (true);

  ... // usage of the value

  jerry_release_value (boolean_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_create_error

**Summary**

Create new JavaScript error object.

**Prototype**

```c
jerry_value_t
jerry_create_error (jerry_error_t error_type,
                    const jerry_char_t *message_p);
```

- `error_type` - type of error
- `message_p` - value of 'message' property of constructed error object
- return value - value of the constructed error object

**Example**

```c
{
  jerry_value_t error_obj = jerry_create_error (JERRY_ERROR_TYPE,
                                                (const jerry_char_t *) "error");

  ... // usage of error_obj


  jerry_release_value (error_obj);
}
```

**See also**

- [jerry_value_has_error_flag](#jerry_value_has_error_flag)
- [jerry_value_clear_error_flag](#jerry_value_clear_error_flag)
- [jerry_value_set_error_flag](#jerry_value_set_error_flag)


## jerry_create_error_sz

**Summary**

Create new JavaScript error object.

**Prototype**

```c
jerry_value_t
jerry_create_error_sz (jerry_error_t error_type,
                       const jerry_char_t *message_p,
                       jerry_size_t message_size);
```

- `error_type` - type of the error
- `message_p` - value of 'message' property of the constructed error object
- `message_size` - size of the message in bytes
- return value - value of the constructed error object

**Example**

```c
{
  const jerry_char_t *message = "error";
  jerry_value_t error_obj = jerry_create_error_sz (JERRY_ERROR_COMMON,
                                                   message,
                                                   strlen ((const char *) message));

  ... // usage of error_obj

  jerry_release_value (error_obj);
}
```

**See also**

- [jerry_create_error](#jerry_create_error)


## jerry_create_external_function

**Summary**

Create an external function object.

**Prototype**

```c
jerry_value_t
jerry_create_external_function (jerry_external_handler_t handler_p);
```

- `handler_p` - pointer to native handler of the function object
- return value - value of the constructed function object

**Example**

```c
static jerry_value_t
handler (const jerry_value_t function_obj,
         const jerry_value_t this_val,
         const jerry_value_t args_p[],
         const jerry_length_t args_cnt)
{
  printf ("native handler called!\n");

  return jerry_create_boolean (true);
}

{
  jerry_value_t func_val = jerry_create_external_function (handler);
  jerry_value_t glob_obj = jerry_get_global_object ();

  // after this, script can invoke the native handler through "handler_field (1, 2, 3);"
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "handler_field");
  // set property and release the return value without any check
  jerry_release_value (jerry_set_property (glob_obj, prop_name, func_val));
  jerry_release_value (prop_name);

  jerry_release_value (func_val);
  jerry_release_value (glob_obj);
}
```

**See also**

- [jerry_external_handler_t](#jerry_external_handler_t)
- [jerry_set_property](#jerry_set_property)
- [jerry_call_function](#jerry_call_function)


## jerry_create_number

**Summary**

Creates a `jerry_value_t` representing a number value.

**Prototype**

```c
jerry_value_t
jerry_create_number (double value);
```

- `value` - double value from which a `jerry_value_t` will be created
- return value - a `jerry_value_t` created from the given double argument

**Example**

```c
{
  jerry_value_t number_value = jerry_create_number (3.14);

  ... // usage of the value

  jerry_release_value (number_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_number_infinity](#jerry_create_number_infinity)
- [jerry_create_number_nan](#jerry_create_number_nan)


## jerry_create_number_infinity

**Summary**

Creates a `jerry_value_t` representing a positive or negative infinity value.

**Prototype**

```c
jerry_value_t
jerry_create_number_infinity (bool sign);
```

- `sign` - true for negative Infinity and false for positive Infinity
- return value - a `jerry_value_t` representing the infinity value

**Example**

```c
{
  jerry_value_t positive_inf_value = jerry_create_number_infinity (false);

  ... // usage of the positive_inf_value

  jerry_release_value (positive_inf_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_number](#jerry_create_number)
- [jerry_create_number_nan](#jerry_create_number_nan)


## jerry_create_number_nan

**Summary**

Creates a `jerry_value_t` representing a not-a-number value.

**Prototype**

```c
jerry_value_t
jerry_create_number_nan (void);
```

- return value - a `jerry_value_t` representing the not-a-number value

**Example**

```c
{
  jerry_value_t nan_value = jerry_create_number_nan ();

  ... // usage of the nan_value

  jerry_release_value (nan_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_number](#jerry_create_number)
- [jerry_create_number_infinity](#jerry_create_number_infinity)


## jerry_create_null

**Summary**

Creates and returns a `jerry_value_t` with type null object.

**Prototype**

```c
jerry_value_t
jerry_create_null (void);
```

- return value - a `jerry_value_t` representing null.

**Example**

```c
{
  jerry_value_t null_value = jerry_create_null ();

  ... // usage of the value

  jerry_release_value (null_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_create_object

**Summary**

Create new JavaScript object, like with new Object().

**Prototype**

```c
jerry_value_t
jerry_create_object (void);
```

- return value - value of the created object

**Example**

```c
{
  jerry_value_t object_value = jerry_create_object ();

  ... // usage of object_value

  jerry_release_value (object_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_create_promise

**Summary**

Create an empty promise object which can be resolved or rejected later
by calling jerry_resolve_or_reject_promise.

*Note*: This API depends on the ES2015-subset profile.

**Prototype**

```c
jerry_value_t
jerry_create_promise (void)
```

- return value - value of the newly created promise

**Example**

```c
{
  jerry_value_t p = jerry_create_promise ();

  ...// usage of the promise

  jerry_release_value (p);
}

**See also**

- [jerry_resolve_or_reject_promise](#jerry_resolve_or_reject_promise)
- [jerry_release_value](#jerry_release_value)


## jerry_create_string

**Summary**

Create string from a valid CESU8 string.

**Prototype**

```c
jerry_value_t
jerry_create_string (const jerry_char_t *str_p);
```

- `str_p` - pointer to string
- return value - value of the created string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value  = jerry_create_string (char_array);

  ... // usage of string_value

  jerry_release_value (string_value);
}
```

**See also**

- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)
- [jerry_create_string_sz](#jerry_create_string_sz)


## jerry_create_string_sz

**Summary**

Create string from a valid CESU8 string.

**Prototype**

```c
jerry_value_t
jerry_create_string_sz (const jerry_char_t *str_p,
                        jerry_size_t str_size)
```

- `str_p` - pointer to string
- `str_size` - size of the string
- return value - value of the created string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value  = jerry_create_string_sz (char_array,
                                                        strlen ((const char *) char_array));

  ... // usage of string_value

  jerry_release_value (string_value);
}

```

**See also**

- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)
- [jerry_create_string](#jerry_create_string)


## jerry_create_string_from_utf8

**Summary**

Create string from a valid UTF8 string.

*Note*: The difference from [jerry_create_string](#jerry_create_string) is that it accepts utf-8 string instead of cesu-8 string.

**Prototype**

```c
jerry_value_t
jerry_create_string_from_utf8 (const jerry_char_t *str_p);
```

- `str_p` - pointer to string
- return value - value of the created string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value  = jerry_create_string_from_utf8 (char_array);

  ... // usage of string_value

  jerry_release_value (string_value);
}
```

**See also**

- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)
- [jerry_create_string_sz_from_utf8](#jerry_create_string_sz_from_utf8)


## jerry_create_string_sz_from_utf8

**Summary**

Create string from a valid UTF8 string.

*Note*: The difference from [jerry_create_string_sz](#jerry_create_string_sz) is that it accepts utf-8 string instead of cesu-8 string.

**Prototype**

```c
jerry_value_t
jerry_create_string_sz (const jerry_char_t *str_p,
                        jerry_size_t str_size)
```

- `str_p` - pointer to string
- `str_size` - size of the string
- return value - value of the created string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value  = jerry_create_string_sz_from_utf8 (char_array,
                                                                  strlen ((const char *) char_array));

  ... // usage of string_value

  jerry_release_value (string_value);
}

```

**See also**

- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)
- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)


## jerry_create_undefined

**Summary**

Creates a `jerry_value_t` representing an undefined value.

**Prototype**

```c
jerry_value_t
jerry_create_undefined (void);
```

- return value - value of undefined

**Example**

```c
{
  jerry_value_t undefined_value = jerry_create_undefined ();

  ... // usage of the value

  jerry_release_value (undefined_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


# General API functions of JS objects

## jerry_has_property

**Summary**

Checks whether the object or its prototype objects have the given property.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_has_property (const jerry_value_t obj_val,
                    const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value - JavaScript boolean value that evaluates to
  - true, if the property exists
  - false, otherwise

**Example**

```c
{
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "handler_field");

  jerry_value_t has_prop_js = jerry_has_property (global_object, prop_name);
  bool has_prop = jerry_get_boolean_value (has_prop_js);

  jerry_release_value (has_prop_js);
  jerry_release_value (prop_name);
  jerry_release_value (global_object);
}
```

**See also**

- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)


## jerry_has_own_property

**Summary**

Checks whether the object has the given property.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_has_own_property (const jerry_value_t obj_val,
                        const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value - JavaScript boolean value that evaluates to
  - true, if the property exists
  - false, otherwise

**Example**

```c
{
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "handler_field");

  jerry_value_t has_prop_js = jerry_has_own_property (global_object, prop_name);
  bool has_prop = jerry_get_boolean_value (has_prop_js);

  jerry_release_value (jas_prop_js);
  jerry_release_value (prop_name);
  jerry_release_value (global_object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_delete_property](#jerry_delete_property)


## jerry_delete_property

**Summary**

Delete a property from an object.

**Prototype**

```c
bool
jerry_delete_property (const jerry_value_t obj_val,
                       const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - true, if property was deleted successfully
  - false, otherwise

**Example**

```c
{
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");

  jerry_delete_property (global_object, prop_name);

  jerry_release_value (prop_name);
  jerry_release_value (global_object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_get_property](#jerry_get_property)


## jerry_get_property

**Summary**

Get value of a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_property (const jerry_value_t obj_val,
                    const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - value of property, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");

  jerry_value_t prop_value = jerry_get_property (obj_val, prop_name);

  jerry_release_value (prop_name);
  jerry_release_value (global_object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_set_property](#jerry_set_property)
- [jerry_get_property_by_index](#jerry_get_property_by_index)
- [jerry_set_property_by_index](#jerry_set_property_by_index)


## jerry_get_property_by_index

**Summary**

Get value by an index from the specified object.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_property_by_index (const jerry_value_t obj_val,
                             uint32_t index);
```

- `obj_val` - object value
- `index` - index number
- return value
  - stored value on the specified index, if success
  - thrown exception, otherwise.

**Example**

```c
{
  jerry_value_t object;

  ... // create or acquire object

  jerry_value_t value = jerry_get_property_by_index (object, 5);

  ...

  jerry_release_value (value);
  jerry_release_value (object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_get_property](#jerry_get_property)
- [jerry_set_property](#jerry_set_property)
- [jerry_set_property_by_index](#jerry_set_property_by_index)


## jerry_set_property

**Summary**

Set a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_set_property (const jerry_value_t obj_val,
                    const jerry_value_t prop_name_val,
                    const jerry_value_t value_to_set)
```

- `obj_val` - object value
- `prop_name_val` - property name
- `value_to_set` - value to set
- return value
  - true, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value_to_set;

  ... // create or acquire value to set

  jerry_value_t glob_obj = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");

  jerry_value_t set_result = jerry_set_property (glob_obj, prop_name, value_to_set);

  ... // check result of property set call

  jerry_release_value (set_result);
  jerry_release_value (prop_name);

  ...

  jerry_release_value (value_to_set);
  jerry_release_value (glob_obj);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_get_property](#jerry_get_property)
- [jerry_get_property_by_index](#jerry_get_property_by_index)
- [jerry_set_property_by_index](#jerry_set_property_by_index)


## jerry_set_property_by_index

**Summary**

Set indexed value in the specified object

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_set_property_by_index (const jerry_value_t obj_val,
                             uint32_t index,
                             const jerry_value_t value_to_set);
```

- `obj_val` - object value
- `index` - index number
- `value_to_set` - value to set
- return value
  - true, if field value was set successfully
  - thrown exception, otherwise

**Example**

```c
{
  jerry_value_t object;
  jerry_value_t value_to_set;

  ... // create or acquire object and value to set

  jerry_value_t ret_val = jerry_set_property_by_index (object, 5, value_to_set);

  ...

  jerry_release_value (value_to_set);
  jerry_release_value (ret_val);
  jerry_release_value (object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_get_property](#jerry_get_property)
- [jerry_set_property](#jerry_set_property)
- [jerry_get_property_by_index](#jerry_get_property_by_index)


## jerry_init_property_descriptor_fields

**Summary**

Initialize property descriptor.

**Prototype**

```c
void
jerry_init_property_descriptor_fields (jerry_property_descriptor_t *prop_desc_p);
```

- `prop_desc_p` - pointer to property descriptor

**Example**

```c
{
  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  ... // usage of prop_desc

  jerry_free_property_descriptor_fields (&prop_desc);
}
```

**See also**

- [jerry_define_own_property](#jerry_define_own_property)
- [jerry_get_own_property_descriptor](#jerry_get_own_property_descriptor)
- [jerry_free_property_descriptor_fields](#jerry_free_property_descriptor_fields)


## jerry_define_own_property

**Summary**

Define a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_define_own_property (const jerry_value_t obj_val,
                           const jerry_value_t prop_name_val,
                           const jerry_property_descriptor_t *prop_desc_p);
```

- `obj_val` - object value
- `prop_name_val` - property name
- `prop_desc_p` - pointer to property descriptor
- return value
  - true, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t global_obj_val = jerry_get_global_object ();

  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  jerry_value_t value_to_set;

  ... // create or acquire value to set

  prop_desc.is_value_defined = true;
  prop_desc.value = value_to_set;

  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");
  jerry_value_t return_value = jerry_define_own_property (global_obj_val, prop_name, &prop_desc);
  jerry_release_value (return_value);
  jerry_release_value (prop_name);

  jerry_free_property_descriptor_fields (&prop_desc);
  jerry_release_value (global_obj_val);
}
```

**See also**

- [jerry_init_property_descriptor_fields](#jerry_init_property_descriptor_fields)
- [jerry_get_own_property_descriptor](#jerry_get_own_property_descriptor)
- [jerry_free_property_descriptor_fields](#jerry_free_property_descriptor_fields)


## jerry_get_own_property_descriptor

**Summary**

Construct property descriptor from specified property.

**Prototype**

```c
bool
jerry_get_own_property_descriptor (const jerry_value_t  obj_val,
                                   const jerry_value_t prop_name_val,
                                   jerry_property_descriptor_t *prop_desc_p);
```

- `obj_val` - object value
- `prop_name_val` - property name
- `prop_desc_p` - pointer to property descriptor
- return value

**Example**

```c
{
  jerry_value_t global_obj_val = jerry_get_global_object ();

  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");
  jerry_get_own_property_descriptor (global_obj_val, prop_name, &prop_desc);
  jerry_release_value (prop_name);

  ... // usage of property descriptor

  jerry_free_property_descriptor_fields (&prop_desc);
  jerry_release_value (global_obj_val);
}
```

**See also**

- [jerry_init_property_descriptor_fields](#jerry_init_property_descriptor_fields)
- [jerry_define_own_property](#jerry_define_own_property)
- [jerry_free_property_descriptor_fields](#jerry_free_property_descriptor_fields)


## jerry_free_property_descriptor_fields

**Summary**

Free fields of property descriptor (setter, getter and value).

**Prototype**

```c
void
jerry_free_property_descriptor_fields (const jerry_property_descriptor_t *prop_desc_p);
```

- `prop_desc_p` - pointer to property descriptor

**Example**

```c
{
  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  ... // usage of property descriptor

  jerry_free_property_descriptor_fields (&prop_desc);
}
```

**See also**

- [jerry_init_property_descriptor_fields](#jerry_init_property_descriptor_fields)
- [jerry_define_own_property](#jerry_define_own_property)
- [jerry_get_own_property_descriptor](#jerry_get_own_property_descriptor)


## jerry_call_function

**Summary**

Call function specified by a function value. Error flag must
not be set for any arguments of this function. Value of `this`
parameter should be set to `undefined` for non-method calls.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_call_function (const jerry_value_t func_obj_val,
                     const jerry_value_t this_val,
                     const jerry_value_t args_p[],
                     jerry_size_t args_count);
```

- `func_obj_val` - the function object to call
- `this_val` - object for 'this' binding
- `args_p` - function's call arguments
- `args_count` - number of arguments
- return value - returned jerry value of the called function

**Example**

```c
{
  jerry_value_t val;

  ... // receiving val

  if (jerry_value_is_function (val))
  {
    jerry_value_t this_val = jerry_create_undefined ();
    jerry_value_t ret_val = jerry_call_function (val, this_val, NULL, 0);

    if (!jerry_value_has_error_flag (ret_val))
    {
      ... // handle return value
    }

    jerry_release_value (ret_val);
    jerry_release_value (this_val);
  }

  jerry_release_value (val);
}
```

**See also**

- [jerry_is_function](#jerry_is_function)
- [jerry_create_external_function](#jerry_create_external_function)


## jerry_construct_object

**Summary**

Construct object, invoking specified function object as constructor.
Error flag must not be set for any arguments of this function.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_construct_object (const jerry_value_t func_obj_val,
                        const jerry_value_t args_p[],
                        jerry_size_t args_count);
```

- `func_obj_val` - function object to call
- `args_p` - function's call arguments
- `args_count` - number of arguments
- return value - returned value of the invoked constructor

**Example**

```c
{
  jerry_value_t val;

  ... // receiving val

  if (jerry_is_constructor (val))
  {
    jerry_value_t ret_val = jerry_construct_object (val, NULL, 0);

    if (!jerry_value_has_error_flag (ret_val))
    {
      ... // handle return value
    }

    jerry_release_value (ret_val);
  }
}
```

**See also**

 - [jerry_is_constructor](#jerry_is_constructor)


## jerry_get_object_keys

**Summary**

Get keys of the specified object value.

**Prototype**

```c
jerry_value_t
jerry_get_object_keys (const jerry_value_t obj_val);
```

- `obj_val` - object value
- return value
  - array object value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t object;
  ... // create or acquire object

  jerry_value_t keys_array = jerry_get_object_keys (object);

  ... // usage of keys_array

  jerry_release_value (keys_array);
}
```

**See also**

- [jerry_get_property](#jerry_get_property)
- [jerry_set_property](#jerry_set_property)


## jerry_get_prototype

**Summary**

Get the prototype of the specified object.

**Prototype**

```c
jerry_value_t
jerry_get_prototype (const jerry_value_t obj_val);
```

- `obj_val` - object value
- return value
  - object value, if success
  - null or thrown error, otherwise

**Example**

```c
{
  jerry_value_t object;
  ... // create or acquire object

  jerry_value_t prototype = jerry_get_prototype (object);

  ... // usage of prototype object

  jerry_release_value (prototype);
  jerry_release_value (object);
}
```

**See also**

- [jerry_set_prototype](#jerry_set_prototype)


## jerry_set_prototype

**Summary**

Set the prototype of the specified object.

**Prototype**

```c
jerry_value_t
jerry_set_prototype (const jerry_value_t obj_val,
                     const jerry_value_t proto_obj_val);
```

- `obj_val` - object value
- `proto_obj_val` - prototype object value
- return value
  - true, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t object;
  jerry_value_t prototype;

  ... // create or acquire object and prototype

  jerry_value_t ret_val = jerry_set_prototype (object, prototype);

  jerry_release_value (ret_val);
  jerry_release_value (prototype);
  jerry_release_value (object);
}
```

**See also**

- [jerry_get_prototype](#jerry_get_prototype)


## jerry_get_object_native_handle

**Summary**

**Deprecated: Please use jerry_get_object_native_pointer instead.**

Get native handle, previously associated with specified object.

**Prototype**

```c
bool
jerry_get_object_native_handle (const jerry_value_t obj_val,
                                uintptr_t *out_handle_p);
```

- `obj_val` - object value
- `out_handle_p` - handle value (output parameter).
- return value
  - true, if there is handle associated with the object
  - false, otherwise

**Example**

```c
{
  jerry_value_t object;
  uintptr_t handle_set;

  ... // receive or construct object and handle_set value

  jerry_set_object_native_handle (object, handle_set, NULL);

  ...

  uintptr_t handle_get;
  bool is_there_associated_handle = jerry_get_object_native_handle (object, &handle_get);
}
```

**See also**

- [jerry_create_object](#jerry_create_object)
- [jerry_set_object_native_handle](#jerry_set_object_native_handle)
- [jerry_get_object_native_pointer](#jerry_get_object_native_pointer)


## jerry_set_object_native_handle

**Summary**

**Deprecated: Please use jerry_set_object_native_pointer instead.**

Set native handle and an optional free callback for the specified object.

*Note*: If native handle was already set for the object, its value is updated.

*Note*: If a non-NULL free callback is specified, it will be called
        by the garbage collector when the object is freed. The free
        callback always overwrites the previous value, so passing
        a NULL value deletes the current free callback.

**Prototype**

```c
void
jerry_set_object_native_handle (const jerry_value_t obj_val,
                                uintptr_t handle_p,
                                jerry_object_free_callback_t freecb_p);
```

- `obj_val` - object value to set handle in
- `handle_p` - handle value
- `freecb_p` - pointer to "free" callback or NULL

**Example**

```c
{
  jerry_value_t object;
  uintptr_t handle_set;

  ... // receive or construct object and handle_set value

  jerry_set_object_native_handle (object, handle_set, NULL);

  ...

  uintptr_t handle_get;
  bool is_there_associated_handle = jerry_get_object_native_handle (object, &handle_get);
}
```

**See also**

- [jerry_create_object](#jerry_create_object)
- [jerry_get_object_native_handle](#jerry_get_object_native_handle)
- [jerry_set_object_native_pointer](#jerry_set_object_native_pointer)


## jerry_get_object_native_pointer

**Summary**

Get native pointer and its type information.
The pointer and the type information are previously associated with the object by jerry_set_object_native_pointer.

*Note*: It is recommended to ensure that the `out_native_info_p` value pointer
        is equal to the native info pointer that is expected, before casting
        and accessing the `out_native_pointer_p`.
        An example of when this is important: external functions that expect
        `this` to have a native pointer of a certain C type.
        It is possible in JavaScript to change `this` at will – using `call()`,
        `apply()` or `bind()`. Therefore, it is possible that the native pointer
        of `this` is not of the expected C type. To handle this safely and
        securely, one must always add type checks to make sure that the
        `out_native_pointer_p` is of the expected type, before casting
        and dereferencing `out_native_pointer_p`.

*Note*: `out_native_pointer_p` and `out_native_info_p` can be NULL, and it means the
        caller doesn't want to get the native_pointer or type infomation.

**Prototype**

```c
bool
jerry_get_object_native_pointer (const jerry_value_t obj_val,
                                 void **out_native_pointer_p,
                                 const jerry_object_native_info_t **out_native_info_p)
```

- `obj_val` - object value to get native pointer from.
- `out_native_pointer_p` - native pointer (output parameter).
- `out_native_info_p` - native pointer's type infomation (output parameter).
- return value
  - true, if there is native pointer associated with the object
  - false, otherwise

**Example**

```c
typedef struct {
  int foo;
  bool bar;
} native_obj_t;

static void native_freecb (void *native_p)
{
  ... // free the native pointer
}

// NOTE: The address (!) of type_info acts as a way to uniquely "identify" the
// C type `native_obj_t *`.
static const jerry_object_native_info_t native_obj_type_info =
{
  .free_cb = native_freecb
};

// Function creating JS object that is "backed" by a native_obj_t *:
{
  ...

  // construct object and native_set value:
  jerry_value_t object = ...;
  native_obj_t *native_obj = malloc(sizeof(*native_obj));
  jerry_set_object_native_pointer (object, native_obj, &native_obj_type_info);

  ...
}

// Native method, `this` is expected to be "backed" by a native_obj_t *:
{
  void *native_p;
  const jerry_object_native_info_t *type_p;
  bool has_p = jerry_get_object_native_pointer (this_val, &native_p, &type_p);

  if (has_p)
  {
    // The type_p pointer address itself is used to identify the type:
    if (type_p == &native_obj_type_info)
    {
      // The type of this's native pointer matches what is expected.
      // Only now is it safe to cast to native_obj_t * and dereference the
      // pointer:
      native_obj_t *native_obj = native_p;
      native_obj->bar = ...; // Safe to access now!
    }
    else
    {
      // The type of this's native pointer is NOT what we expected!
      // We should not cast to native_obj_t * and dereference because it's unsafe.
      // Handle the error here, for example throw an error.
    }
  }
  ...
}
```

**See also**

- [jerry_create_object](#jerry_create_object)
- [jerry_set_object_native_pointer](#jerry_set_object_native_pointer)
- [jerry_object_native_info_t](#jerry_object_native_info_t)


## jerry_set_object_native_pointer

**Summary**

Set native pointer and an optional type information for the specified object.
You can get them by calling jerry_get_object_native_pointer later.

*Note*: If native pointer was already set for the object, its value is updated.

*Note*: If a non-NULL free callback is specified in the native type information,
        it will be called by the garbage collector when the object is freed.
        The type info is always overwrites the previous value, so passing
        a NULL value deletes the current type info.

**Prototype**

```c
bool
jerry_set_object_native_pointer (const jerry_value_t obj_val,
                                 void *native_p,
                                 const jerry_object_native_info_t *info_p)
```

- `obj_val` - object to set native pointer in.
- `native_p` - native pointer.
- `info_p` - native pointer's type infomation or NULL. When used, this should
             be a long-lived pointer, usually a pointer to a
             `static const jerry_object_native_info_t` makes most sense.

**Example**

See [jerry_get_object_native_pointer](#jerry_get_object_native_pointer) for a
best-practice example.

**See also**

- [jerry_create_object](#jerry_create_object)
- [jerry_get_object_native_pointer](#jerry_get_object_native_pointer)
- [jerry_object_native_info_t](#jerry_object_native_info_t)


## jerry_foreach_object_property

**Summary**

Applies the given function to every property in the given object.

**Prototype**

```c
bool
jerry_foreach_object_property (jerry_value_t obj_val,
                               jerry_object_property_foreach_t foreach_p,
                               void *user_data_p);
```

- `obj_val` - object value
- `foreach_p` - foreach function, that will be applied for each property
- `user_data_p` - user data for foreach function
- return value
  - true, if object fields traversal was performed successfully, i.e.:
    - no unhandled exceptions were thrown in object fields traversal
    - object fields traversal was stopped on callback that returned false
  - false, otherwise

**Example**

```c
bool foreach_function (const jerry_value_t prop_name,
                       const jerry_value_t prop_value,
                       void *user_data_p)
{

  ... // implementation of the foreach function

}

{
  jerry_value_t object;
  ... // receive or construct object

  double data = 3.14; // example data

  jerry_foreach_object_property (object, foreach_function, &data);

}
```

**See also**

- [jerry_object_property_foreach_t](#jerry_object_property_foreach_t)


# Input validator functions

## jerry_is_valid_utf8_string

**Summary**

Validate UTF-8 string.

**Prototype**

```c
bool
jerry_is_valid_utf8_string (const jerry_char_t *utf8_buf_p, /**< UTF-8 string */
                            jerry_size_t buf_size) /**< string size */
```

- `utf8_buf_p` - UTF-8 input string
- `buf_size` - input string size

**Example**

```c
{
  const jerry_char_t script[] = "print ('Hello, World!');";
  size_t script_size = strlen ((const char *) script);

  if (jerry_is_valid_utf8_string (script, (jerry_size_t) script_size))
  {
    jerry_run_simple (script, script_size, JERRY_INIT_EMPTY);
  }
}
```

**See also**

- [jerry_run_simple](#jerry_run_simple)
- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)
- [jerry_create_string_sz_from_utf8](#jerry_create_string_sz_from_utf8)
- [jerry_get_utf8_string_size](#jerry_get_utf8_string_size)
- [jerry_get_utf8_string_length](#jerry_get_utf8_string_length)
- [jerry_string_to_utf8_char_buffer](#jerry_string_to_utf8_char_buffer)
- [jerry_substring_to_utf8_char_buffer](#jerry_substring_to_utf8_char_buffer)

## jerry_is_valid_cesu8_string

**Summary**

Validate CESU-8 string.

**Prototype**

```c
bool
jerry_is_valid_cesu8_string (const jerry_char_t *cesu8_buf_p, /**< CESU-8 string */
                             jerry_size_t buf_size) /**< string size */
```

- `cesu8_buf_p` - CESU-8 input string
- `buf_size` - input string size

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "Hello, World!";
  size_t script_size = strlen ((const char *) script);

  if (jerry_is_valid_cesu8_string (script, (jerry_size_t) script_size))
  {
    jerry_value_t string_value = jerry_create_string_sz (script,
                                                         (jerry_size_t) script_size));

    ... // usage of string_value

    jerry_release_value (string_value);
  }

  jerry_cleanup ();
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_create_string_sz](#jerry_create_string_sz)
- [jerry_get_string_size](#jerry_get_string_size)
- [jerry_get_string_length](#jerry_get_string_length)
- [jerry_string_to_char_buffer](#jerry_string_to_char_buffer)
- [jerry_substring_to_char_buffer](#jerry_substring_to_char_buffer)


# Snapshot functions

## jerry_parse_and_save_snapshot

**Summary**

Generate snapshot from the specified source code.

**Prototype**

```c
size_t
jerry_parse_and_save_snapshot (const jerry_char_t *source_p,
                               size_t source_size,
                               bool is_for_global,
                               bool is_strict,
                               uint32_t *buffer_p,
                               size_t buffer_size);
```

- `source_p` - script source, it must be a valid utf8 string.
- `source_size` - script source size, in bytes.
- `is_for_global` - snapshot would be executed as global (true) or eval (false).
- `is_strict` - strict mode
- `buffer_p` - buffer to save snapshot to.
- `buffer_size` - the buffer's size.
- return value
  - the size of snapshot, if it was generated succesfully (i.e. there are no syntax errors in source
    code, buffer size is sufficient, and snapshot support is enabled in current configuration through
    JERRY_ENABLE_SNAPSHOT_SAVE)
  - 0 otherwise.

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  static uint32_t global_mode_snapshot_buffer[256];
  const jerry_char_t *code_to_snapshot_p = "(function () { return 'string from snapshot'; }) ();";

  size_t global_mode_snapshot_size = jerry_parse_and_save_snapshot (code_to_snapshot_p,
                                                                    strlen ((const char *) code_to_snapshot_p),
                                                                    true,
                                                                    false,
                                                                    global_mode_snapshot_buffer,
                                                                    sizeof (global_mode_snapshot_buffer) / sizeof (uint32_t));

  jerry_cleanup ();
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_exec_snapshot](#jerry_exec_snapshot)


## jerry_exec_snapshot

**Summary**

Execute snapshot from the specified buffer.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_exec_snapshot (const uint32_t *snapshot_p,
                     size_t snapshot_size,
                     bool copy_bytecode);
```

- `snapshot_p` - pointer to snapshot
- `snapshot_size` - size of snapshot
- `copy_bytecode` - flag, indicating whether the passed snapshot buffer should be copied to the
   engine's memory. If set the engine should not reference the buffer after the function returns
   (in this case, the passed buffer could be freed after the call). Otherwise (if the flag is not
   set) - the buffer could only be freed after the engine stops (i.e. after call to jerry_cleanup).
- return value
  - result of bytecode, if run was successful
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t res;
  static uint32_t global_mode_snapshot_buffer[256];
  const jerry_char_t *code_to_snapshot_p = "(function () { return 'string from snapshot'; }) ();";

  jerry_init (JERRY_INIT_EMPTY);
  size_t global_mode_snapshot_size = jerry_parse_and_save_snapshot (code_to_snapshot_p,
                                                                    strlen ((const char *) code_to_snapshot_p),
                                                                    true,
                                                                    false,
                                                                    global_mode_snapshot_buffer,
                                                                    sizeof (global_mode_snapshot_buffer) / sizeof (uint32_t));
  jerry_cleanup ();

  jerry_init (JERRY_INIT_EMPTY);

  res = (jerry_exec_snapshot (global_mode_snapshot_buffer,
                              global_mode_snapshot_size,
                              false);

  jerry_cleanup ();
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_parse_and_save_snapshot](#jerry_parse_and_save_snapshot)


## jerry_parse_and_save_literals

**Summary**

Collect the used literals from the given source code and save them into a specific file in a list or C format.
These literals are generated by the parser, they are valid identifiers and none of them are magic string.

**Prototype**

```c
size_t
jerry_parse_and_save_literals (const jerry_char_t *source_p,
                               size_t source_size,
                               bool is_strict,
                               uint32_t *buffer_p,
                               size_t buffer_size,
                               bool is_c_format);
```

- `source_p` - script source, it must be a valid utf8 string.
- `source_size` - script source size, in bytes.
- `is_strict` - strict mode.
- `buffer_p` - buffer to save literals to.
- `buffer_size` - the buffer's size.
- `is_c_format` - the output format would be C-style (true) or a simple list (false).
- return value
  - the size of the literal-list, if it was generated succesfully (i.e. the list of literals isn't empty,
    and literal-save support is enabled in current configuration through JERRY_ENABLE_SNAPSHOT_SAVE)
  - 0 otherwise.

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  static uint32_t save_literal_buffer[256];
  const jerry_char_t *code_for_literal_save_p = "var obj = { a:'aa', bb:'Bb' }";

  size_t literal_sizes = jerry_parse_and_save_literals (code_for_literal_save_p,
                                                        strlen ((const char *) code_for_literal_save_p),
                                                        false,
                                                        save_literal_buffer,
                                                        sizeof (save_literal_buffer) / sizeof (uint32_t),
                                                        true);

  if (literal_sizes != 0)
  {
    FILE *literal_file_p = fopen ("literals.txt", "w");
    fwrite (save_literal_buffer, sizeof (uint8_t), literal_sizes, literal_file_p);
    fclose (literal_file_p);
  }

  jerry_cleanup ();
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_register_magic_strings](#jerry_register_magic_strings)


# Miscellaneous functions

## jerry_set_vm_exec_stop_callback

**Summary**

When JERRY_FEATURE_VM_EXEC_STOP is enabled a callback function can be
specified by this function. This callback is periodically called when
JerryScript executes an ECMAScript program.

If the callback returns with undefined value the ECMAScript execution
continues. Otherwise the result is thrown by the engine (if the error
flag is not set for the returned value the engine automatically sets
it). The callback function might be called again even if it threw
an error. In this case the function must throw the same error again.

To reduce the CPU overhead of constantly checking the termination
condition the callback is called when a backward jump is executed
or an exception is caught. Setting the `frequency` to a greater
than `1` value reduces this overhead further. If its value is N
only every Nth event (backward jump, etc.) trigger the next check.


**Prototype**

```c
void
jerry_set_vm_exec_stop_callback (jerry_vm_exec_stop_callback_t stop_cb,
                                 void *user_p,
                                 uint32_t frequency);
```

- `stop_cb` - periodically called callback (passing NULL disables this feature)
- `user_p` - user pointer passed to the `stop_cb` function
- `frequency` - frequency of calling the `stop_cb` function

**Example**

```c
static jerry_value_t
vm_exec_stop_callback (void *user_p)
{
  static int countdown = 10;

  while (countdown > 0)
  {
    countdown--;
    return jerry_create_undefined ();
  }

  // The error flag is added automatically.
  return jerry_create_string ((const jerry_char_t *) "Abort script");
}

{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_set_vm_exec_stop_callback (vm_exec_stop_callback, &countdown, 16);

  // Inifinte loop.
  const char *src_p = "while(true) {}";

  jerry_value_t src = jerry_parse ((jerry_char_t *) src_p, strlen (src_p), false);
  jerry_release_value (jerry_run (src));
  jerry_release_value (src);
  jerry_cleanup ();
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_parse](#jerry_parse)
- [jerry_run](#jerry_run)
- [jerry_vm_exec_stop_callback_t](#jerry_vm_exec_stop_callback_t)
