//roardmx.h:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2019
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#ifndef _LIBROARLIGHT_ROARDMX_H_
#define _LIBROARLIGHT_ROARDMX_H_

#include "libroarlight.h"

#define ROAR_ROARDMX_VERSION        0x00

#define ROAR_ROARDMX_MASK_FLAGS     0xF0
#define ROAR_ROARDMX_MASK_TYPE      0x0F

#define ROAR_ROARDMX_DATA_LENGTH    ((1<<(sizeof(char)*8))-1) /* 255 */

// we have a offset of one nibble, see ROAR_ROARDMX_MASK_FLAGS
#define ROAR_ROARDMX_FLAG_PACKED    0x10

#define ROAR_ROARDMX_TYPE_SSET      0x00 /* simple set */
#define ROAR_ROARDMX_TYPE_IPO1      0x01 /* linear interploation */
#define ROAR_ROARDMX_TYPE_IPOLIN    ROAR_ROARDMX_TYPE_IPO1
#define ROAR_ROARDMX_TYPE_IPO4      0x02 /* poly5 interpolation (a*t^4 + b*t^3 + c*t^2 + d*t + e) */
#define ROAR_ROARDMX_TYPE_INC8S     0x03 /* signed 8 bit increment */
#define ROAR_ROARDMX_TYPE_RANGESET  0x04 /* like simple set but set ranges of channels */
#define ROAR_ROARDMX_TYPE_EVENT     0x0E /* used to transmit simple events like step, blackout and stuff */
#define ROAR_ROARDMX_TYPE_CONTROL   0x0F /* Used to transmit other control information like codec version */

// Events:
// event types (ETYPE) will be bit or-ed with event.
#define ROAR_ROARDMX_MASK_EVENT     0x3F
#define ROAR_ROARDMX_MASK_ETYPE     0xC0

#define ROAR_ROARDMX_ETYPE_BEAT     0x00 /* event happens NOW */
#define ROAR_ROARDMX_ETYPE_OFF      0x40 /* event ended */
#define ROAR_ROARDMX_ETYPE_ON       0x80 /* event started */
#define ROAR_ROARDMX_ETYPE_HOLD     0xC0 /* event is still running, used as keep-alive */

// events: must be in range 0x00 to 0x3F.
#define ROAR_ROARDMX_EVENT_NONE     0x00 /* can be used as keep-alive or padding */
#define ROAR_ROARDMX_EVENT_STEP     0x01
#define ROAR_ROARDMX_EVENT_TAP      0x02
#define ROAR_ROARDMX_EVENT_BEAT     0x03 /* audio signal generated beat */
#define ROAR_ROARDMX_EVENT_BLACKOUT 0x04 /* set all light channels to zero */
#define ROAR_ROARDMX_EVENT_FULLON   0x05 /* set all light channels to full intensity */
#define ROAR_ROARDMX_EVENT_FLASH    0x06 /* like fullon but can also use strobe. */
#define ROAR_ROARDMX_EVENT_STROBE   0x07 /* triggers all strobes */
#define ROAR_ROARDMX_EVENT_STROBEREADY 0x08 /* strobe is ready */
#define ROAR_ROARDMX_EVENT_STROBELOAD  0x09 /* strobe is loading */
#define ROAR_ROARDMX_EVENT_FOG      0x0A /* triggers all fog machines */
#define ROAR_ROARDMX_EVENT_FOGREADY 0x0B /* Fog machine is ready */
#define ROAR_ROARDMX_EVENT_FOGHEAT  0x0C /* Fog machine is heating up */
/* hole */
#define ROAR_ROARDMX_EVENT_NOTIFY0  0x10 /* Notify e.g. IRC highlight, ringing phone, ... */
#define ROAR_ROARDMX_EVENT_NOTIFY1  0x11
#define ROAR_ROARDMX_EVENT_NOTIFY2  0x12
#define ROAR_ROARDMX_EVENT_NOTIFY3  0x13
/* hole */
#define ROAR_ROARDMX_EVENT_USER0    0x30 /* User defined event #0  */
#define ROAR_ROARDMX_EVENT_USER1    0x31 /* User defined event #1  */
#define ROAR_ROARDMX_EVENT_USER2    0x32 /* User defined event #2  */
#define ROAR_ROARDMX_EVENT_USER3    0x33 /* User defined event #3  */
#define ROAR_ROARDMX_EVENT_USER4    0x34 /* User defined event #4  */
#define ROAR_ROARDMX_EVENT_USER5    0x35 /* User defined event #5  */
#define ROAR_ROARDMX_EVENT_USER6    0x36 /* User defined event #6  */
#define ROAR_ROARDMX_EVENT_USER7    0x37 /* User defined event #7  */
#define ROAR_ROARDMX_EVENT_USER8    0x38 /* User defined event #8  */
#define ROAR_ROARDMX_EVENT_USER9    0x39 /* User defined event #9  */
#define ROAR_ROARDMX_EVENT_USER10   0x3A /* User defined event #10 */
#define ROAR_ROARDMX_EVENT_USER11   0x3B /* User defined event #11 */
#define ROAR_ROARDMX_EVENT_USER12   0x3C /* User defined event #12 */
#define ROAR_ROARDMX_EVENT_USER13   0x3D /* User defined event #13 */
#define ROAR_ROARDMX_EVENT_USER14   0x3E /* User defined event #14 */
#define ROAR_ROARDMX_EVENT_USER15   0x3F /* User defined event #15 */

/* subcommands for ROAR_ROARDMX_TYPE_CONTROL: */
#define ROAR_ROARDMX_CTL_VERSION    0x00 /* Version of Codec */
#define ROAR_ROARDMX_CTL_SYNC       0x01 /* Sync state */

/* Flags for ROAR_ROARDMX_CTL_SYNC: */
#define ROAR_ROARDMX_SYNC_DIRTY     0x01 /* global state is not sync, some more data needs to be send first */
#define ROAR_ROARDMX_SYNC_CLEAN     0x02 /* global state is sync */
#define ROAR_ROARDMX_SYNC_SOF       0x03 /* start of frame */
#define ROAR_ROARDMX_SYNC_EOF       0x04 /* end of frame */
#define ROAR_ROARDMX_SYNC_CLOCK     0x80 /* clock tick */

// Data format on the wire:
/*
 * All packets consist of a header and a body.
 * The header has the size of 3 Bytes:
 * 0) version. Must be ROAR_ROARDMX_VERSION.
 * 1) packet type and flags.
 * 2) length of body in bytes.
 *
 * the type-flags byte consists of the packet type in the lower nibble and
 * the flags in the upper nibble.
 *
 * the header is followd by a body of zero to 255 bytes length.
 * the content of the body depends on the used type of packet.
 *
 * SSET: (simple set) Sets channels to values.
 *  The body consists of {channel, value}-pairs.
 *  Every entry is 3 byte: hi byte of channel, low byte of channel, value.
 *
 * IPO1: not implemented.
 * IPO4: not implemented.
 * INC8S: not implemented.
 * RANGESET: Sets ranges of channels.
 *  The body consists of {start, stop, value}-trippels.
 *  Every entry is 5 byte: hi byte of start channel, low byte of start channel,
 *  hi byte of end channel, low byte of end channel and value.
 *
 * EVENT: transmit simple yet abstract events.
 * The body is an array of bytes. Each byte represents one event.
 * Each event has the upper two bits set to the meta type of the event
 * (beat, on, off, hold) and the lower 6 bits to the type of te event.
 *
 * CONTROL: transmit other control information.
 * The body consists of a version byte followed by a command byte.
 * The version byte is always zero by this standard.
 * The command byte is one of ROAR_ROARDMX_CTL_*
 * The rest of body depends on both the version and command byte.
 * For command=VERSION packets the rest of the body is a single byte with the value
 * of zero as of this standard.
 * For command=SYNC packets the rest of the body is a single byte which is a bitarray
 * for flags defined above as ROAR_ROARDMX_SYNC_*.
 */

struct roar_roardmx_message {
 unsigned char version;
 unsigned char flags;
 unsigned char type;
 size_t        length;
 unsigned char data[3 /* header */ + ROAR_ROARDMX_DATA_LENGTH /* data */];
};

// database access:
int roar_roardmx_str2event(const char * event);
const char * roar_roardmx_event2str(const int event);

// generic things:
int roar_roardmx_message_new (struct roar_roardmx_message * mes);

// low level:
//int roar_roardmx_message_set_flag(struct roar_roardmx_message * mes, unsigned char   flag);
//int roar_roardmx_message_set_len (struct roar_roardmx_message * mes, size_t          type);
//int roar_roardmx_message_get_data(struct roar_roardmx_message * mes, unsigned char ** data);

// medium level:
int roar_roardmx_message_set_type(struct roar_roardmx_message * mes, unsigned char   type);
int roar_roardmx_message_get_flag(struct roar_roardmx_message * mes, unsigned char * flag);
int roar_roardmx_message_get_type(struct roar_roardmx_message * mes, unsigned char * type);
int roar_roardmx_message_get_len (struct roar_roardmx_message * mes, size_t        * length);


// IO:
int roar_roardmx_message_send(struct roar_roardmx_message * mes, struct roar_vio_calls * vio);
int roar_roardmx_message_recv(struct roar_roardmx_message * mes, struct roar_vio_calls * vio);

// Data/high level:
// * *:
int roar_roardmx_message_numchannels(struct roar_roardmx_message * mes); // also works for rangesets
int roar_roardmx_message_add_chanval(struct roar_roardmx_message * mes, uint16_t   channel, unsigned char   val);
int roar_roardmx_message_get_chanval(struct roar_roardmx_message * mes, uint16_t * channel, unsigned char * val, int index);

// * SSET:
int roar_roardmx_message_new_sset   (struct roar_roardmx_message * mes);

// * IPO1:
// Not yet supported.
// * IPO4:
// Not yet supported.
// * INC8S:
// Not yet supported.
// * RANGESET:
int roar_roardmx_message_new_rangeset(struct roar_roardmx_message * mes);
int roar_roardmx_message_add_rangeval(struct roar_roardmx_message * mes, uint16_t   start, uint16_t   end, unsigned char   val);
int roar_roardmx_message_get_rangeval(struct roar_roardmx_message * mes, uint16_t * start, uint16_t * end, unsigned char * val, int index);

// * EVENT:
int roar_roardmx_message_new_event(struct roar_roardmx_message * mes);
int roar_roardmx_message_add_events(struct roar_roardmx_message * mes, const uint8_t * events, size_t len);
#define roar_roardmx_message_add_event(mes,event) roar_roardmx_message_add_events((mes), &(uint8_t){(event)}, 1)
int roar_roardmx_message_get_events(struct roar_roardmx_message * mes, const uint8_t ** events, size_t * len);

// * CONTROL:
int roar_roardmx_message_new_version(struct roar_roardmx_message * mes, uint8_t version);
int roar_roardmx_message_new_sync(struct roar_roardmx_message * mes, uint8_t flags);
int roar_roardmx_message_get_subcommand(struct roar_roardmx_message * mes);
int roar_roardmx_message_get_ctl_version(struct roar_roardmx_message * mes);
int roar_roardmx_message_get_ctl_flags(struct roar_roardmx_message * mes);

#endif

//ll
