// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/views/web_apps/web_app_uninstall_dialog_view.h"
#include <memory>

#include "base/bind.h"
#include "base/bind_helpers.h"
#include "base/macros.h"
#include "base/metrics/histogram_macros.h"
#include "base/strings/string_util.h"
#include "base/strings/utf_string_conversions.h"
#include "chrome/browser/ui/browser_dialogs.h"
#include "chrome/browser/ui/native_window_tracker.h"
#include "chrome/browser/ui/views/chrome_layout_provider.h"
#include "chrome/browser/ui/views/chrome_typography.h"
#include "chrome/browser/ui/views/frame/browser_view.h"
#include "chrome/browser/ui/views/web_apps/web_app_info_image_source.h"
#include "chrome/browser/web_applications/components/install_finalizer.h"
#include "chrome/browser/web_applications/web_app_icon_manager.h"
#include "chrome/browser/web_applications/web_app_provider.h"
#include "chrome/grit/chromium_strings.h"
#include "chrome/grit/generated_resources.h"
#include "components/constrained_window/constrained_window_views.h"
#include "components/strings/grit/components_strings.h"
#include "components/url_formatter/elide_url.h"
#include "content/public/browser/clear_site_data_utils.h"
#include "extensions/browser/extension_dialog_auto_confirm.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/gfx/geometry/insets.h"
#include "ui/gfx/image/image_skia.h"
#include "ui/views/controls/button/checkbox.h"
#include "ui/views/controls/image_view.h"
#include "ui/views/controls/label.h"
#include "ui/views/layout/box_layout.h"
#include "ui/views/view.h"
#include "ui/views/widget/widget.h"
#include "ui/views/window/dialog_delegate.h"

namespace {

constexpr int kIconSizeInDip = 32;

}  // namespace

WebAppUninstallDialogDelegateView::WebAppUninstallDialogDelegateView(
    Profile* profile,
    WebAppUninstallDialogViews* dialog_view,
    web_app::AppId app_id,
    std::map<SquareSizePx, SkBitmap> icon_bitmaps)
    : dialog_(dialog_view), app_id_(app_id), profile_(profile) {
  auto* provider = web_app::WebAppProvider::Get(profile_);
  DCHECK(provider);

  app_name_ = base::UTF8ToUTF16(provider->registrar().GetAppShortName(app_id_));

  app_launch_url_ = provider->registrar().GetAppLaunchURL(app_id_);
  DCHECK(!app_launch_url_.is_empty());
  DCHECK(app_launch_url_.is_valid());

  gfx::Size image_size{kIconSizeInDip, kIconSizeInDip};

  image_ = gfx::ImageSkia(
      std::make_unique<WebAppInfoImageSource>(kIconSizeInDip, icon_bitmaps),
      image_size);

  DialogDelegate::SetButtonLabel(
      ui::DIALOG_BUTTON_OK,
      l10n_util::GetStringUTF16(IDS_EXTENSION_PROMPT_UNINSTALL_BUTTON));

  ChromeLayoutProvider* layout_provider = ChromeLayoutProvider::Get();
  SetLayoutManager(std::make_unique<views::BoxLayout>(
      views::BoxLayout::Orientation::kVertical, gfx::Insets(),
      layout_provider->GetDistanceMetric(
          views::DISTANCE_RELATED_CONTROL_VERTICAL)));

  // modify by xiaohuyang, Set the new style of 'WebAppUninstallDialogDelegateView', 2020/11/30 --start
  // Set new margins and add title.
#if 0
  // Add margins for the icon plus the icon-title padding so that the dialog
  // contents align with the title text.
  gfx::Insets insets =
      layout_provider->GetDialogInsetsForContentType(views::TEXT, views::TEXT);
  set_margins(insets + gfx::Insets(0, insets.left() + kIconSizeInDip, 0, 0));

#else
  gfx::Insets margin_insets = layout_provider->GetDialogInsetsForContentType(
      views::CONTROL, views::CONTROL);
  set_margins(gfx::Insets(0, margin_insets.left(), 10, margin_insets.right()));

  // title
  auto title_label = std::make_unique<views::Label>(
      l10n_util::GetStringFUTF16(IDS_EXTENSION_PROMPT_UNINSTALL_TITLE,
                                    app_name_), 
      CONTEXT_BODY_TEXT_LARGE);
  title_label->SetMultiLine(true);
  title_label->SetLineHeight(20);
  title_label->SetFontList(gfx::FontList().DeriveWithWeight(gfx::Font::Weight::BOLD));
  title_label->SetHorizontalAlignment(gfx::ALIGN_CENTER);
  title_label->SizeToFit(
      layout_provider->GetDistanceMetric(
          ChromeDistanceMetric::DISTANCE_BUBBLE_PREFERRED_WIDTH) -
      margins().width());
  AddChildView(std::move(title_label));
#endif
  // modify by xiaohuyang, Set the new style of 'WebAppUninstallDialogDelegateView', 2020/11/30 --end

  base::string16 checkbox_label = l10n_util::GetStringFUTF16(
      IDS_EXTENSION_UNINSTALL_PROMPT_REMOVE_DATA_CHECKBOX,
      url_formatter::FormatUrlForSecurityDisplay(
          app_launch_url_, url_formatter::SchemeDisplay::OMIT_CRYPTOGRAPHIC));

  auto checkbox = std::make_unique<views::Checkbox>(checkbox_label);
  checkbox->SetMultiLine(true);
  checkbox_ = AddChildView(std::move(checkbox));

  chrome::RecordDialogCreation(chrome::DialogIdentifier::EXTENSION_UNINSTALL);

  ProcessAutoConfirmValue();
}

WebAppUninstallDialogDelegateView::~WebAppUninstallDialogDelegateView() {
  if (dialog_)
    dialog_->CallCallback(/*uninstalled=*/false);
}

bool WebAppUninstallDialogDelegateView::Accept() {
  if (!dialog_)
    return true;

  bool uninstalled = Uninstall();
  if (checkbox_->GetChecked())
    ClearWebAppSiteData();

  dialog_->CallCallback(uninstalled);
  dialog_ = nullptr;
  return true;
}

bool WebAppUninstallDialogDelegateView::Cancel() {
  if (dialog_) {
    dialog_->CallCallback(/*uninstalled=*/false);
    dialog_ = nullptr;
  }
  return true;
}

gfx::Size WebAppUninstallDialogDelegateView::CalculatePreferredSize() const {
  const int width = ChromeLayoutProvider::Get()->GetDistanceMetric(
                        DISTANCE_MODAL_DIALOG_PREFERRED_WIDTH) -
                    margins().width();
  return gfx::Size(width, GetHeightForWidth(width));
}

ui::ModalType WebAppUninstallDialogDelegateView::GetModalType() const {
  return ui::MODAL_TYPE_WINDOW;
}

base::string16 WebAppUninstallDialogDelegateView::GetWindowTitle() const {
  // modify by xiaohuyang, Do not dispaly window title.
#if 0
  return l10n_util::GetStringFUTF16(IDS_EXTENSION_PROMPT_UNINSTALL_TITLE,
                                    app_name_);
#else
  return base::string16();
#endif
}

gfx::ImageSkia WebAppUninstallDialogDelegateView::GetWindowIcon() {
  return image_;
}

bool WebAppUninstallDialogDelegateView::ShouldShowWindowIcon() const {
  return true;
}

bool WebAppUninstallDialogDelegateView::ShouldShowCloseButton() const {
  // modify by xiaohuyang, Dispaly close button.
#if 0
  return false;
#else
  return true;
#endif
}

bool WebAppUninstallDialogDelegateView::Uninstall() {
  auto* provider = web_app::WebAppProvider::Get(profile_);
  DCHECK(provider);

  if (!provider->install_finalizer().CanUserUninstallExternalApp(app_id_))
    return false;

  dialog_->UninstallStarted();

  provider->install_finalizer().UninstallExternalAppByUser(app_id_,
                                                           base::DoNothing());
  return true;
}

void WebAppUninstallDialogDelegateView::ClearWebAppSiteData() {
  content::ClearSiteData(
      base::BindRepeating(
          [](content::BrowserContext* browser_context) {
            return browser_context;
          },
          base::Unretained(profile_)),
      url::Origin::Create(app_launch_url_), /*clear_cookies=*/true,
      /*clear_storage=*/true, /*clear_cache=*/true,
      /*avoid_closing_connections=*/false, base::DoNothing());
}

void WebAppUninstallDialogDelegateView::ProcessAutoConfirmValue() {
  switch (extensions::ScopedTestDialogAutoConfirm::GetAutoConfirmValue()) {
    case extensions::ScopedTestDialogAutoConfirm::NONE:
      break;
    case extensions::ScopedTestDialogAutoConfirm::ACCEPT_AND_OPTION:
      checkbox_->SetChecked(/*checked=*/true);
      AcceptDialog();
      break;
    case extensions::ScopedTestDialogAutoConfirm::ACCEPT:
      AcceptDialog();
      break;
    case extensions::ScopedTestDialogAutoConfirm::CANCEL:
      CancelDialog();
      break;
  }
}

WebAppUninstallDialogViews::WebAppUninstallDialogViews(Profile* profile,
                                                       gfx::NativeWindow parent)
    : parent_(parent), profile_(profile) {
  if (parent)
    parent_window_tracker_ = NativeWindowTracker::Create(parent);
}

WebAppUninstallDialogViews::~WebAppUninstallDialogViews() {
  if (view_)
    view_->CancelDialog();
}

void WebAppUninstallDialogViews::ConfirmUninstall(
    const web_app::AppId& app_id,
    WebAppUninstallDialogViews::OnWebAppUninstallDialogClosed closed_callback) {
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  // TODO(loyso): Add UMA_HISTOGRAM_ENUMERATION here.

  app_id_ = app_id;
  closed_callback_ = std::move(closed_callback);

  if (parent_ && parent_window_tracker_->WasNativeWindowClosed()) {
    CallCallback(/*uninstalled=*/false);
    return;
  }

  auto* provider = web_app::WebAppProvider::Get(profile_);
  DCHECK(provider);

  registrar_observer_.Add(&provider->registrar());

  provider->icon_manager().ReadAllIcons(
      app_id, base::BindOnce(&WebAppUninstallDialogViews::OnAllIconsRead,
                             weak_ptr_factory_.GetWeakPtr()));
}

void WebAppUninstallDialogViews::SetDialogShownCallbackForTesting(
    base::OnceClosure callback) {
  dialog_shown_callback_for_testing_ = std::move(callback);
}

void WebAppUninstallDialogViews::OnAllIconsRead(
    std::map<SquareSizePx, SkBitmap> icon_bitmaps) {
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);

  if (parent_ && parent_window_tracker_->WasNativeWindowClosed()) {
    CallCallback(/*uninstalled=*/false);
    return;
  }

  view_ = new WebAppUninstallDialogDelegateView(profile_, this, app_id_,
                                                std::move(icon_bitmaps));

  constrained_window::CreateBrowserModalDialogViews(view_, parent_)->Show();

  if (dialog_shown_callback_for_testing_)
    std::move(dialog_shown_callback_for_testing_).Run();
}

void WebAppUninstallDialogViews::OnWebAppUninstalled(
    const web_app::AppId& app_id) {
  // Handle the case when web app was uninstalled externally and we have to
  // cancel current dialog.
  if (app_id == app_id_ && view_)
    view_->CancelDialog();
}

void WebAppUninstallDialogViews::OnAppRegistrarDestroyed() {
  registrar_observer_.RemoveAll();
  if (view_)
    view_->CancelDialog();
}

void WebAppUninstallDialogViews::UninstallStarted() {
  // Next OnWebAppUninstalled should be ignored. Unsubscribe:
  registrar_observer_.RemoveAll();
}

void WebAppUninstallDialogViews::CallCallback(bool uninstalled) {
  view_ = nullptr;
  std::move(closed_callback_).Run(uninstalled);
}

// static
std::unique_ptr<web_app::WebAppUninstallDialog>
web_app::WebAppUninstallDialog::Create(Profile* profile,
                                       gfx::NativeWindow parent) {
  return std::make_unique<WebAppUninstallDialogViews>(profile, parent);
}
