/*
 * Copyright (C) 2024 Uniontech Technology Co., Ltd.
 *
 * Author:     liuzheng <liuzheng@uniontech.com>
 *
 * Maintainer: liuzheng <liuzheng@uniontech.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef CLIPBOARD_H
#define CLIPBOARD_H

#include "displayjack_clipboardstruct.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C"
{
#endif

// listener
typedef void (*ClipDataAddListenerPtr)(int, void *);
typedef void (*ClipDataDeleteListenerPtr)(const char *, void *);
typedef void (*ClipDataStateListenerPtr)(int, void *);

/**
 * @brief ClipHandle结构体
 * 用于对剪切板存储目录保存和回调函数地址存储的保存
 */
struct ClipHandle {
    void *backend;
    char *xdgSessionType;
    bool isWayland;
    void *addUserData;
    void *deleteUserData;
    void *stateUserData;
    char *pathDir;

    // 监听改变信号
    ClipDataAddListenerPtr ClipDataAddListener;
    ClipDataDeleteListenerPtr ClipDataDeleteListener;
    ClipDataStateListenerPtr ClipDataStateListener;
};
//自定义ClipHandle指针
typedef struct ClipHandle *ClipHandlePtr;

//导出ClipHandlePtr
extern ClipHandlePtr pClipHandle;

/**
 * @brief init_clip_board 初始化剪切板
 * @param path 数据存储和加载目录
 * @return 无
 */
int init_clipboard(const char *path);

/**
 * @brief init_clipboard_connect 初始化剪贴板连接
 * @param path 无
 * @return 无
 * @note 此函数如果调用了LoadClipData函数可不调用此函数
 */
void init_clipboard_connect();

/**
 * @brief destroy_clipboard 销毁剪贴板
 * @param 无
 * @return 无
 */
void destroy_clipboard();

/**
 * @brief load_clipdata 加载剪贴板数据
 * @param 无
 * @return 无
 */
int load_clipdata();

/**
 * @brief get_clipboard_data_property_at 获取剪贴板数据属性
 * @param clipIndex 剪切板数据索引
 * @return 剪切板属性数据
 */
ClipboardDataProperty get_clipboard_data_property_at(int clipIndex);

/**
 * @brief get_clipdata_count 获取剪贴板数据数量
 * @param 无
 * @return 剪切板数目
 */
int get_clipdata_count();

/**
 * @brief remove_cilpboard_at 根据索引移除剪贴板数据
 * @param clipIndex 剪切板索引
 * @return 无
 */
void remove_cilpboard_at(int clipIndex);

/**
 * @brief move_cilpboard_at 移动剪贴板数据
 * @param sourceClipIndex 当前索引
 * @param targetClipIndex 目标索引
 * @return 是否移动成功
 */
int move_cilpboard_at(int sourceClipIndex, int targetClipIndex);

/**
 * @brief get_clip_index_by_name 根据名称获取剪贴板索引
 * @param name 剪切板数据名字
 * @return 索引
 */
int get_clip_index_by_name(const char *name);

/**
 * @brief set_clip_on_top 设置剪贴板置顶
 * @param index 索引
 * @return 无
 */
void set_clip_on_top(int index);

/**
 * @brief cilpboard_clear 清空剪贴板
 * @param 无
 * @return 无
 */
void cilpboard_clear();

/**
 * @brief get_clipboard_data_record_at 获取剪贴板数据记录
 * @param clipIndex 剪切板数据索引
 * @param dataIndex 剪切板数据记录索引
 * @return 剪切板实际数据
 */
ClipboardDataRecord get_clipboard_data_record_at(int clipIndex, int dataIndex);

/**
 * @brief get_clipboard_data_record_count 获取剪贴板数据记录数量
 * @param clipIndex 剪切板数据索引
 * @return 剪切板数据条目
 */
int get_clipboard_data_record_count(int clipIndex);

/**
 * @brief remove_clipboard_data_record_at 移除剪贴板数据记录
 * @param clipIndex 剪切板数据索引
 * @param dataIndex 剪切板数据记录索引
 * @return 无
 */
void remove_clipboard_data_record_at(int clipIndex, int dataIndex);

/**
 * @brief add_clipboard_data 添加剪贴板数据
 * @param propety 属性数据
 * @param record 记录数据
 * @return 无
 */
void add_clipboard_data(ClipboardDataProperty *propety, ClipboardDataRecord *record);

/**
 * @brief set_clipboard_num 设置剪贴板数据数目
 * @param num 数目
 * @return 无
 */
void set_clipboard_num(int num);

/**
 * @brief do_clipdata_add_listener_callback 添加剪贴板数据监听回调
 * @param func 回调函数地址
 * @param data 对象
 * @return 无
 */
void do_clipdata_add_listener_callback(ClipDataAddListenerPtr func, void *data);

/**
 * @brief do_clipdata_delete_listener_callback 移除剪贴板数据监听回调
 * @param func 回调函数地址
 * @param data 对象
 * @return 无
 */
void do_clipdata_delete_listener_callback(ClipDataDeleteListenerPtr func, void *data);

/**
 * @brief do_clipdata_state_listener_callback 数据传输状态监听回调
 * @param func 回调函数地址
 * @param data 对象
 * @return 无
 */
void do_clipdata_state_listener_callback(ClipDataStateListenerPtr func, void *data);

/**
 * @brief get_clipboard_data_record_info_at 获取剪贴板数据记录
 * @param clipIndex 剪切板数据索引
 * @param dataIndex 剪切板数据记录索引
 * @return 剪切板实际数据
 */
ClipDataRecordInfo get_clipboard_data_record_info_at(int clipIndex, int dataIndex);

/**
 * @brief get_clipboard_data_property_info_at 获取剪贴板数据属性
 * @param clipIndex 剪切板数据索引www
 * @return 剪切板属性数据
 */
ClipDataPropertyInfo get_clipboard_data_property_info_at(int clipIndex);

#ifdef __cplusplus
}
#endif
#endif
