// SPDX-License-Identifier: Apache-2.0
#include <crispy/StrongLRUCache.h>
#include <crispy/utils.h>

#include <catch2/catch_test_macros.hpp>

#include <iostream>
#include <string_view>

using namespace crispy;
using namespace std;
using namespace std::string_view_literals;

// NOLINTBEGIN(misc-const-correctness,readability-function-cognitive-complexity)
TEST_CASE("strong_lru_cache.operator_index", "")
{
    auto cache = strong_lru_cache<int, string_view>(strong_hashtable_size { 8 }, lru_capacity { 4 });

    cache[1] = "1"sv;
    REQUIRE(cache[1] == "1"sv);
    REQUIRE(joinHumanReadable(cache.keys()) == "1");

    cache[2] = "2"sv;
    REQUIRE(cache[2] == "2"sv);
    REQUIRE(joinHumanReadable(cache.keys()) == "2, 1");

    cache[3] = "3"sv;
    REQUIRE(cache[3] == "3"sv);
    REQUIRE(joinHumanReadable(cache.keys()) == "3, 2, 1");

    cache[4] = "4"sv;
    REQUIRE(cache[4] == "4"sv);
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    cache[5] = "5"sv;
    REQUIRE(cache[5] == "5"sv);
    REQUIRE(joinHumanReadable(cache.keys()) == "5, 4, 3, 2");

    cache[6] = "6"sv;
    REQUIRE(cache[6] == "6"sv);
    REQUIRE(joinHumanReadable(cache.keys()) == "6, 5, 4, 3");
}

TEST_CASE("strong_lru_cache.at", "")
{
    auto cache = strong_lru_cache<int, string>(strong_hashtable_size { 8 }, lru_capacity { 4 });
    for (int i = 1; i <= 4; ++i)
        cache[i] = std::to_string(i);
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    CHECK_THROWS_AS(cache.at(-1), std::out_of_range);
    CHECK_NOTHROW(cache.at(1));
}

TEST_CASE("strong_lru_cache.clear", "[lrucache]")
{
    auto cache = strong_lru_cache<int, string>(strong_hashtable_size { 8 }, lru_capacity { 4 });
    for (int i = 1; i <= 4; ++i)
        cache[i] = std::to_string(i);
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    CHECK(cache.size() == 4);
    cache.clear();
    CHECK(cache.size() == 0);
}

TEST_CASE("strong_lru_cache.touch", "")
{
    auto cache = strong_lru_cache<int, string>(strong_hashtable_size { 8 }, lru_capacity { 4 });
    for (int i = 1; i <= 4; ++i)
        cache[i] = std::to_string(i);
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // no-op (not found)
    cache.touch(-1);
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // no-op (found)
    cache.touch(4);
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // middle to front
    cache.touch(3);
    REQUIRE(joinHumanReadable(cache.keys()) == "3, 4, 2, 1");

    // back to front
    cache.touch(1);
    REQUIRE(joinHumanReadable(cache.keys()) == "1, 3, 4, 2");
}

TEST_CASE("strong_lru_cache.contains", "")
{
    auto cache = strong_lru_cache<int, string>(strong_hashtable_size { 8 }, lru_capacity { 4 });
    for (int i = 1; i <= 4; ++i)
        cache[i] = std::to_string(i);
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // not found: no-op
    REQUIRE(!cache.contains(-1));
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // found: front is no-op
    REQUIRE(cache.contains(4));
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // found: middle to front
    REQUIRE(cache.contains(3));
    REQUIRE(joinHumanReadable(cache.keys()) == "3, 4, 2, 1");

    // found: back to front
    REQUIRE(cache.contains(1));
    REQUIRE(joinHumanReadable(cache.keys()) == "1, 3, 4, 2");
}

TEST_CASE("strong_lru_cache.try_emplace", "")
{
    auto cache = strong_lru_cache<int, int>(strong_hashtable_size { 4 }, lru_capacity { 2 });

    auto rv = cache.try_emplace(2, [](auto) { return 4; });
    CHECK(rv);
    CHECK(joinHumanReadable(cache.keys()) == "2");
    CHECK(cache.at(2) == 4);

    rv = cache.try_emplace(3, [](auto) { return 6; });
    CHECK(rv);
    CHECK(joinHumanReadable(cache.keys()) == "3, 2");
    CHECK(cache.at(2) == 4);
    CHECK(cache.at(3) == 6);

    rv = cache.try_emplace(2, [](auto) { return -1; });
    CHECK_FALSE(rv);
    CHECK(joinHumanReadable(cache.keys()) == "2, 3");
    CHECK(cache.at(2) == 4);
    CHECK(cache.at(3) == 6);
}

TEST_CASE("strong_lru_cache.try_get", "")
{
    auto cache = strong_lru_cache<int, string>(strong_hashtable_size { 8 }, lru_capacity { 4 });
    for (int i = 1; i <= 4; ++i)
        cache[i] = std::to_string(i);
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // no-op (not found)
    REQUIRE(cache.try_get(-1) == nullptr);
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // no-op (found)
    auto* const p1 = cache.try_get(4);
    REQUIRE(p1 != nullptr);
    REQUIRE(*p1 == "4");
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // middle to front
    auto* const p2 = cache.try_get(3);
    REQUIRE(p2 != nullptr);
    REQUIRE(*p2 == "3");
    REQUIRE(joinHumanReadable(cache.keys()) == "3, 4, 2, 1");

    // back to front
    auto* const p3 = cache.try_get(1);
    REQUIRE(p3 != nullptr);
    REQUIRE(*p3 == "1");
    REQUIRE(joinHumanReadable(cache.keys()) == "1, 3, 4, 2");
}

TEST_CASE("strong_lru_cache.get_or_emplace", "[lrucache]")
{
    auto cache = strong_lru_cache<int, int>(strong_hashtable_size { 4 }, lru_capacity { 2 });

    int const& a = cache.get_or_emplace(2, [](auto) { return 4; });
    CHECK(a == 4);
    CHECK(cache.at(2) == 4);
    CHECK(cache.size() == 1);
    CHECK(joinHumanReadable(cache.keys()) == "2"sv);

    int const& a2 = cache.get_or_emplace(2, [](auto) { return -4; });
    CHECK(a2 == 4);
    CHECK(cache.at(2) == 4);
    CHECK(cache.size() == 1);

    int const& b = cache.get_or_emplace(3, [](auto) { return 6; });
    CHECK(b == 6);
    CHECK(cache.at(3) == 6);
    CHECK(cache.size() == 2);
    CHECK(joinHumanReadable(cache.keys()) == "3, 2"sv);

    int const& c = cache.get_or_emplace(4, [](auto) { return 8; });
    CHECK(joinHumanReadable(cache.keys()) == "4, 3"sv);
    CHECK(c == 8);
    CHECK(cache.at(4) == 8);
    CHECK(cache.size() == 2);
    CHECK(cache.contains(3));
    CHECK_FALSE(cache.contains(2)); // thrown out

    int const& b2 = cache.get_or_emplace(3, [](auto) { return -3; });
    CHECK(joinHumanReadable(cache.keys()) == "3, 4"sv);
    CHECK(b2 == 6);
    CHECK(cache.at(3) == 6);
    CHECK(cache.size() == 2);
}

TEST_CASE("strong_lru_cache.remove", "")
{
    auto cache = strong_lru_cache<int, string>(strong_hashtable_size { 8 }, lru_capacity { 4 });
    for (int i = 1; i <= 4; ++i)
        cache[i] = std::to_string(i);
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // remove at head
    cache.remove(4);
    REQUIRE(joinHumanReadable(cache.keys()) == "3, 2, 1");

    // remove in middle
    cache.remove(2);
    REQUIRE(joinHumanReadable(cache.keys()) == "3, 1");

    // remove at tail
    cache.remove(1);
    REQUIRE(joinHumanReadable(cache.keys()) == "3");

    // remove last
    cache.remove(3);
    REQUIRE(joinHumanReadable(cache.keys()).empty());
}

// clang-format off
struct colliding_hasher
{
    strong_hash operator()(int v) noexcept
    {
        // Since the hashtable lookup only looks at the
        // least significant 32 bit, this will always cause
        // a hash-table entry collision.
        return strong_hash { 0, 0, static_cast<uint32_t>(v), 0 };
    }
};
// clang-format on

TEST_CASE("strong_lru_cache.insert_with_cache_collision", "")
{
    auto cache =
        strong_lru_cache<int, int, colliding_hasher>(strong_hashtable_size { 8 }, lru_capacity { 4 });

    cache[1] = 1;
    REQUIRE(joinHumanReadable(cache.keys()) == "1");

    cache[2] = 2;
    REQUIRE(joinHumanReadable(cache.keys()) == "2, 1");

    cache[3] = 3;
    REQUIRE(joinHumanReadable(cache.keys()) == "3, 2, 1");

    cache[4] = 4;
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");
}

TEST_CASE("strong_lru_cache.remove_with_hashTable_lookup_collision", "")
{
    auto cache =
        strong_lru_cache<int, int, colliding_hasher>(strong_hashtable_size { 8 }, lru_capacity { 4 });
    for (int i = 1; i <= 4; ++i)
        cache[i] = 2 * i;
    REQUIRE(joinHumanReadable(cache.keys()) == "4, 3, 2, 1");

    // remove at head
    cache.remove(4);
    REQUIRE(joinHumanReadable(cache.keys()) == "3, 2, 1");

    // remove in middle
    cache.remove(2);
    REQUIRE(joinHumanReadable(cache.keys()) == "3, 1");

    // remove at tail
    cache.remove(1);
    REQUIRE(joinHumanReadable(cache.keys()) == "3");

    // remove last
    cache.remove(3);
    REQUIRE(joinHumanReadable(cache.keys()).empty());
}
// NOLINTEND(misc-const-correctness,readability-function-cognitive-complexity)
