/*
 * Copyright (C) 2020 Uniontech Technology Co., Ltd.
 *
 * Author:     xinbo wang <wangxinbo@uniontech.com>
 *
 * Maintainer: xinbo wang <wangxinbo@uniontech.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef DTK_UTILS_H
#define DTK_UTILS_H

#include <math.h>
#include <stddef.h>
#include <inttypes.h>
#include <stdarg.h>

#ifdef  __cplusplus
extern "C" {
#endif

/** Visibility attribute */
#if defined(__GNUC__) && __GNUC__ >= 4
#define DTK_EXPORT __attribute__ ((visibility("default")))
#else
#define DTK_EXPORT
#endif

/** Deprecated attribute */
#if defined(__GNUC__) && __GNUC__ >= 4
#define DTK_DEPRECATED __attribute__ ((deprecated))
#else
#define DTK_DEPRECATED
#endif

/**
 * \class dtk_array
 *
 * Dynamic array
 *
 * A dtk_array is a dynamic array that can only grow until released. It is
 * intended for relatively small allocations whose size is variable or not known
 * in advance. While construction of a dtk_array does not require all elements to
 * be of the same size, dtk_array_for_each() does require all elements to have
 * the same type and size.
 *
 */
struct dtk_array {
    /** Array size */
    size_t size;
    /** Allocated space */
    size_t alloc;
    /** Array data */
    void *data;
};

/**
 * Initializes the array.
 *
 * \param array Array to initialize
 *
 * \memberof dtk_array
 */
void
dtk_array_init(struct dtk_array *array);

/**
 * Releases the array data.
 *
 * \note Leaves the array in an invalid state.
 *
 * \param array Array whose data is to be released
 *
 * \memberof dtk_array
 */
void
dtk_array_release(struct dtk_array *array);

/**
 * Increases the size of the array by \p size bytes.
 *
 * \param array Array whose size is to be increased
 * \param size Number of bytes to increase the size of the array by
 *
 * \return A pointer to the beginning of the newly appended space, or NULL when
 *         resizing fails.
 *
 * \memberof dtk_array
 */
void *
dtk_array_add(struct dtk_array *array, size_t size);

/**
 * Copies the contents of \p source to \p array.
 *
 * \param array Destination array to copy to
 * \param source Source array to copy from
 *
 * \return 0 on success, or -1 on failure
 *
 * \memberof dtk_array
 */
int
dtk_array_copy(struct dtk_array *array, struct dtk_array *source);

/**
 * Iterates over an array.
 *
 * This macro expresses a for-each iterator for dtk_array. It assigns each
 * element in the array to \p pos, which can then be referenced in a trailing
 * code block. \p pos must be a pointer to the array element type, and all
 * array elements must be of the same type and size.
 *
 * \param pos Cursor that each array element will be assigned to
 * \param array Array to iterate over
 *
 * \relates dtk_array
 * \sa wl_list_for_each()
 */
#define dtk_array_for_each(pos, array, step)					\
    for (pos = (array)->data;					\
         (const char *) pos < ((const char *) (array)->data + (array)->size); \
         (pos)+=(step))

#ifdef  __cplusplus
}
#endif

#endif
