#!/bin/bash

set -e
set -o pipefail

GIT=${GIT:="git"}

repoRoot="$("$GIT" rev-parse --show-toplevel)"
cd "$repoRoot"/tools/codegen

QUICKTYPE=${QUICKTYPE:="npx quicktype"}

if [ "$QUICKTYPE" == "npx quicktype" ]; then
	npm install .
fi

generate() {
	schema="$1"
	shift
	toplevel_type="$1"
	shift
	namespace="$1"
	shift
	include="$1"
	shift
	output_path="$1"

	if [[ ! -f "$schema" ]]; then
		echo "$schema not found" || exit 255
	fi

	filename="/dev/null"

	while read -r line; do
		if [[ $line != //\ stdout &&
			$line != //*.hpp &&
			$line != \#include\ \"*\" ]]; then
			echo "$line" >>"$filename"
			continue
		fi

		if [[ $line == \#include\ \"*\" ]]; then
			header=${line#\#include\ \"}
			header=${header%\"}
			printf "#include \"%s/%s\"\n" \
				"$output_path" \
				"$header" \
				>>"$filename"
			continue
		fi

		echo "// clang-format on" >>"$filename"

		filename="${repoRoot}/$include/$output_path/${line#\/\/ }"
		if [[ $line == //\ stdout ]]; then
			filename="/dev/null"
		fi

		# echo "generating $filename"

		mkdir -p "$(dirname "$filename")"
		{
			echo "// Thish file is generated by /tools/codegen"
			echo "// DO NOT EDIT IT."
			echo ""
			echo "// clang-format off"
		} >"$filename"
	done < <($QUICKTYPE "$schema" \
		--lang c++ \
		-s schema \
		-t "$toplevel_type" \
		--namespace "$namespace" \
		--code-format with-struct \
		--source-style multi-source \
		--include-location global-include \
		--type-style pascal-case-upper-acronyms \
		--member-style camel-case-upper-acronyms \
		--enumerator-style pascal-case-upper-acronyms \
		--no-boost \
		--hide-null-optional)

	{
		echo ""
		echo "// clang-format on"
	} >>"$filename"
}

include="libs/runtime/include"

generate \
	"$repoRoot/runtime-spec/schema/config-schema.json" \
	Config \
	"ocppi::runtime::config::types" \
	"$include" \
	"ocppi/runtime/config/types"

generate \
	"$repoRoot/runtime-spec/schema/state-schema.json" \
	State \
	"ocppi::runtime::state::types" \
	"$include" \
	"ocppi/runtime/state/types"

generate \
	"$repoRoot/runtime-spec/schema/features-schema.json" \
	Features \
	"ocppi::runtime::features::types" \
	"$include" \
	"ocppi/runtime/features/types"

PATCH_FILE=${PATCH_FILE:="$repoRoot"/tools/codegen/fix-unknow-types.patch}

if [[ ! -f "$PATCH_FILE" ]]; then
	exit
fi

cd "$repoRoot"
cp -r "$repoRoot/$include" "$repoRoot/$include.orig"
patch -s -p0 <"$PATCH_FILE"
rm -r "$repoRoot/$include.orig"
