
#include "BackupModule.h"
#include "utils/Utils.h"
#include "utils/global.h"
#include "utils/Device.h"
#include <QString>
#include <QObject>
#include <QFile>



BackupModule::BackupModule(FrameProxyInterface *frame, ComDeepinDaemonUosrecoveryInterface *interface,
                           QObject *parent)
    : QObject(parent),
      ModuleInterface(frame, interface)
{

}

BackupModule::~BackupModule()
{

}

void BackupModule::initialize()
{
    if (m_backupWidget == nullptr) {
        m_backupWidget = new BackupWidget();
    }

    connect(m_backupWidget, &BackupWidget::notifySystemBackup, this, &BackupModule::onSystemBackup);
    connect(m_backupWidget, &BackupWidget::notifyDataBackup, this, &BackupModule::onDataBackup);
    connect(m_recoveryInterface, &ComDeepinDaemonUosrecoveryInterface::ReportProgress,
            this, &BackupModule::updateProgress);
    connect(m_recoveryInterface, &ComDeepinDaemonUosrecoveryInterface::Error,
            this, &BackupModule::onError);
    connect(m_recoveryInterface, &ComDeepinDaemonUosrecoveryInterface::ReportSpace,
            this, &BackupModule::onSpaceChanged);
    connect(m_recoveryInterface, &ComDeepinDaemonUosrecoveryInterface::ReportCheckSpace,
            this, &BackupModule::onReportCheckSpace);

#ifdef UI_TEST
    m_timer = new QTimer(this);
    m_timer->setInterval(100);
    connect(m_timer, &QTimer::timeout, this, &BackupModule::onTimeout);
#endif
}

QString BackupModule::name() const
{
    return "BackupModule";
}

void BackupModule::active()
{
    m_frameProxy->popAllWidget();
    m_frameProxy->setCurrentWidget(this, m_backupWidget);
}

QString BackupModule::icons() const
{
    return QString(":/resources/icons/backup_module.png");
}

QString BackupModule::text() const
{
    return QString(tr("Backup"));
}

void BackupModule::onShowProgress(const QString &mainTitle, const QString &subTitle, const QString &warning)
{
    if (m_progressWidget == nullptr) {
        m_progressWidget = new ProgressWidget(mainTitle, subTitle, warning);
    } else {
        m_progressWidget->setMainTitle(mainTitle);
        m_progressWidget->setSubTitle(subTitle);
        m_progressWidget->setWarning(warning);
    }
    m_progressWidget->setValue(0);
    m_frameProxy->setCurrentWidget(this, m_progressWidget);
    m_frameProxy->enableBackWard(false);
    m_frameProxy->enableModule(false);
    m_progressWidget->start();
}

void BackupModule::updateProgress(const QString &progress)
{

    QJsonObject jsonObject = Utils::QStringToJson(progress);
    int operateType = jsonObject.value("operateType").toInt();
    if ((operateType != OperateType::SystemBackup) && (operateType != OperateType::UserDataBackup)) {
        return;
    }

    if (m_progressWidget != nullptr) {
        auto curProgress = jsonObject.value("progress").toInt();
        auto remainSecond = jsonObject.value("remainSecond").toInt();
        m_progressWidget->setRemainTime(remainSecond);
        m_progressWidget->setValue(curProgress);
        if (curProgress >= 100) {
            onShowResult(true, operateType);
        }
    }
}

void BackupModule::onShowResult(bool success, int operateType, const QString &errorMsg)
{
    if (operateType != SystemBackup && operateType != UserDataBackup) {
        return;
    }

    QString resultText = success ? tr("Backup successful!") : tr("Sorry, backup failed!");
    QString btnText = tr("OK", "button");

    m_frameProxy->setMenuDisabled(false);
    m_frameProxy->setWindowFuncClose(true);

    if (m_progressWidget != nullptr) {
        m_frameProxy->popWidget();
    }

    if (m_resultWidget == nullptr) {
        m_resultWidget = new ResultWidget(success, resultText, errorMsg, btnText);
        connect(m_resultWidget, &ResultWidget::done, this, &BackupModule::onBackHome);
    } else {
        m_resultWidget->set(success, resultText, errorMsg, btnText);
    }

    m_frameProxy->setCurrentWidget(this, m_resultWidget);
    m_frameProxy->enableBackWard(true);
    m_frameProxy->enableModule(true);
}

void BackupModule::onSystemBackup()
{
    if (!Utils::authorization()) {
        return;
    }

    if (m_systemBackupWidget == nullptr) {
        m_systemBackupWidget = new SystemBackupWidget();
        if (m_systemSyncType == RecoveryType::OSTree) {
            m_systemBackupWidget->setDestBtnEnabled(false);
        }

        connect(m_systemBackupWidget, &SystemBackupWidget::cancel, this, &BackupModule::onBack);
        connect(m_systemBackupWidget, &SystemBackupWidget::start, this,
                &BackupModule::onStartSystemBackup);
        connect(m_systemBackupWidget, &SystemBackupWidget::notifySetDestPartition, [=] {
            m_frameProxy->showSettingDialog();
        });
        connect(m_systemBackupWidget, &SystemBackupWidget::notifyBackupManage, [=] {
           m_frameProxy->showBackupFileManagerDialog();
        });
    }

    auto rootUUIDReply = m_recoveryInterface->GetRootUUID();
    rootUUIDReply.waitForFinished();
    if (!rootUUIDReply.isValid()) {
        qCritical() << rootUUIDReply.error();
        onShowResult(false, SystemBackup,
                     tr("DBus error, please try again"));
        return;
    }

    auto destUUIDReply = m_recoveryInterface->GetBackupDeviceUUID(m_frameProxy->rootUUID());
    destUUIDReply.waitForFinished();
    if (!destUUIDReply.isValid()) {
        qCritical() << destUUIDReply.error();
        onShowResult(false, SystemBackup,
                     tr("DBus error, please try again"));
        return;
    }

    m_destUUID = destUUIDReply.value();
    if (!m_destUUID.isEmpty()) {
    //    qInfo() << "destUUID=" << m_destUUID;
        auto partitionReply = m_recoveryInterface->GetPartition(m_destUUID);
        partitionReply.waitForFinished();
        if (!partitionReply.isValid()) {
            qCritical() << "GetPartition failed, err: "<<partitionReply.error();
            return;
        }
        QString partitionInfo = partitionReply.value();
        m_systemBackupWidget->setDestPartitionText(Utils::QStringToJson(partitionInfo));
    }
    m_systemBackupWidget->setNotes("");

    m_frameProxy->setCurrentWidget(this, m_systemBackupWidget);
}

void BackupModule::setSystemBackupSpaceTips(const QString &tips)
{
    if (m_systemBackupWidget != nullptr) {
        m_systemBackupWidget->setTips(tips);
    }
}

void BackupModule::onStartSystemBackup(const QString &remark)
{
    m_curSysBackupReq.clear();
    m_frameProxy->setMenuDisabled(true);
    m_frameProxy->setWindowFuncClose(false);

    SystemBackupRequest request;
    request.username = Utils::getUserName();
    request.rootUUID = m_frameProxy->rootUUID();
    request.destUUID = m_destUUID;
    request.remark = remark;
    QJsonObject jsonObject = request.marshal();

    m_systemBackupWidget->setTips("");
    auto checkSpaceReply = m_recoveryInterface->CheckIncSystemBackupSpace(request.destUUID);
    checkSpaceReply.waitForFinished();
    if (!checkSpaceReply.isValid()) {
        qCritical() << Q_FUNC_INFO << " checkSpaceReply isvalid!";
        onShowResult(false, SystemBackup,
                     tr("DBus error, please try again"));
        return;
    }

    m_curSysBackupReq = Utils::JsonToQString(jsonObject);
}

void BackupModule::onDataBackup()
{
    if (!Utils::checkCommonUserAuthentication()) {
        return;
    }

    if (m_userDataBackupSelectWidget == nullptr) {
        m_userDataBackupSelectWidget = new UserDataBackupSelectWidget();
        connect(m_userDataBackupSelectWidget, &UserDataBackupSelectWidget::cancel, this, &BackupModule::onBack);
        connect(m_userDataBackupSelectWidget, &UserDataBackupSelectWidget::next,
                this, &BackupModule::onCheckSpace);
    }

    m_userDataBackupSelectWidget->setTips("");

    //获取家目录文件
    QString userPath = QStandardPaths::writableLocation(QStandardPaths::HomeLocation);
    QDir userDir(userPath);
    m_fileItems.clear();
    auto fileInfoList = userDir.entryInfoList(QDir::Files | QDir::Hidden | QDir::Dirs | QDir::NoDotAndDotDot, QDir::Type| QDir::DirsFirst);
    for (auto &fileInfo : fileInfoList) {
        FileItem fileItem;
        fileItem.fileName = fileInfo.fileName();
        fileItem.size = fileInfo.size();
        qInfo() << fileInfo.absoluteFilePath();
        if (fileInfo.isDir()) {
            fileItem.fileType = FileType::Dir;
            QDir subDir(fileInfo.absoluteFilePath());
            auto subFileInfoList = subDir.entryInfoList(QDir::Files | QDir::Hidden | QDir::Dirs | QDir::NoDotAndDotDot, QDir::Type| QDir::DirsFirst);
            for (auto &subFileInfo : subFileInfoList) {
                qInfo() << subFileInfo.absoluteFilePath();
                FileItem childFileItem;
                childFileItem.fileName = subFileInfo.fileName();
                childFileItem.size = subFileInfo.size();
                if (subFileInfo.isDir()) {
                    childFileItem.fileType = FileType::Dir;
                } else {
                    childFileItem.fileType = FileType::File;
                }
                fileItem.children.append(childFileItem);
            }
        } else {
            fileItem.fileType = FileType::File;
        }
        m_fileItems.append(fileItem);
    }
    m_userDataBackupSelectWidget->setFiles(m_fileItems);

    auto partitionReply = m_recoveryInterface->ListPartition(true);
    partitionReply.waitForFinished();
    if (!partitionReply.isValid()) {
        qCritical() << partitionReply.error();
        onShowResult(false, UserDataBackup,
                     tr("DBus error, please try again"));
        return;
    }

    m_userDataBackupSelectWidget->setDestDevice(Utils::QStringToJson(partitionReply.value()));
    m_frameProxy->setCurrentWidget(this, m_userDataBackupSelectWidget);
}

QStringList BackupModule::getUsrDataSpecialExcludes()
{
    QStringList excludes = {
            ".config/browser/Default/virtual"
    };

    return excludes;
}

void BackupModule::onStartDataBackup(const QString &remark)
{
    if (m_userDataBackupWidget == nullptr) {
        qCritical() << "m_userDataBackupWidget is nullptr";
        return;
    }

    UserDataBackupRequest request;
    request.username = Utils::getUserName();
    request.remark = remark;
    request.destUUID = m_destUUID;
    if (!m_destUUID.isEmpty()) {
        m_frameProxy->setMenuDisabled(true);
        m_frameProxy->setWindowFuncClose(false);

        request.rootUUID = m_frameProxy->rootUUID();
        request.exclude = m_excludes;
        QJsonObject jsonObject = request.marshal();
        auto requestReply = m_recoveryInterface->UserDataBackup(Utils::JsonToQString(jsonObject));
        requestReply.waitForFinished();
        if (!requestReply.isValid()) {
            qCritical() << requestReply.error();
            onShowResult(false, UserDataBackup, tr("DBus error, please try again"));
            return;
        }
        if (requestReply.value() != OK) {
            onShowResult(false, OperateType::UserDataBackup);
            return;
        }
        onShowProgress(tr("Backing up..."),
                       tr("Time remaining: "),
                       tr("To avoid data loss, please do not use your computer during the process."));
    }
}

void BackupModule::onBack()
{
    if (m_frameProxy) {
        m_frameProxy->back();
    }
}

void BackupModule::onBackHome()
{
    if (nullptr != m_frameProxy) {
        m_frameProxy->backHome();
    }
}

void BackupModule::updateDestPartition(const QJsonObject &jsonObject)
{
    if (m_systemBackupWidget != nullptr) {
        m_systemBackupWidget->setDestPartitionText(jsonObject);
    }

    if (jsonObject.contains("uuid")) {
        m_destUUID = jsonObject.value("uuid").toString();
    }
}

#ifdef UI_TEST
void BackupModule::onTimeout()
{
    m_progressWidget->setValue(++m_progress);
    if (m_progress > 50) {
        m_timer->stop();
        m_progress = 0;
        onShowResult(false, SystemBackup, "test");
    }
}


#endif

void BackupModule::onError(const QString &errMsg)
{
    QJsonObject jsonObject = Utils::QStringToJson(errMsg);
    QString errInfo = "Unknown";
    if (jsonObject.contains("errMsg")) {
        errInfo = jsonObject.value("errMsg").toString();
    }

    int opType = -1;
    if (jsonObject.contains("operateType")) {
        opType = jsonObject.value("operateType").toInt();
    }

    onShowResult(false, opType, errInfo);
}

void BackupModule::setSystemSyncType(int type)
{
    m_systemSyncType = type;
}

void BackupModule::setUserDataSyncType(int type)
{
    m_userDataSyncType = type;
}

void BackupModule::onSpaceChanged(const QString &space)
{
    QJsonObject jsonObject = Utils::QStringToJson(space);
    int operateType = jsonObject.value("operateType").toInt();
    if (operateType == OperateType::CheckUserDataBackupSpace) {
        m_frameProxy->setMenuDisabled(false);
        m_frameProxy->setWindowFuncClose(true);
        m_userDataBackupSelectWidget->setNextBtnEnabled(true);
        m_userDataBackupSelectWidget->stopSpinner();

        bool hasSpace = false;
        if (jsonObject.contains("hasSpace")) {
            hasSpace = jsonObject.value("hasSpace").toBool();
        }

        if (!hasSpace) {
            m_userDataBackupSelectWidget->setTips(tr("Insufficient disk space on the device"));
            m_userDataBackupSelectWidget->setTipsStyleSheet("QLabel {"
                                                            "color: #FF5736;"
                                                            "}");
            return;
        }

        m_userDataBackupSelectWidget->setTips("");
        QString comboBoxTex = m_userDataBackupSelectWidget->getCurComboBoxText();
        this->showUserDataBackupWidget(comboBoxTex);
    }
}

void BackupModule::onCheckSpace()
{
    m_destUUID = m_userDataBackupSelectWidget->getDestDeviceUUID();
    QString destMountPoint;
    if(!Device::getMountPointByUUID(m_destUUID, destMountPoint)) {
        qWarning()<<"onCheckSpace call getMountPointByUUID failed, m_destUUID = "<<m_destUUID;
        return;
    }

    if (Device::isDeviceMountedReadOnly(destMountPoint)) {
        m_userDataBackupSelectWidget->setTips(tr("Cannot back up data to the read-only device"));
        m_userDataBackupSelectWidget->setTipsStyleSheet("QLabel {"
                                                        "color: #FF5736;"
                                                        "}");
        return;
    }

    UserDataBackupRequest request;
    request.username = Utils::getUserName();
    m_excludeFileItems = m_userDataBackupSelectWidget->getExclude();
    request.destUUID = m_destUUID;
    if (!m_destUUID.isEmpty()) {
        m_frameProxy->setMenuDisabled(true);
        m_frameProxy->setWindowFuncClose(false);
        m_userDataBackupSelectWidget->setNextBtnEnabled(false);

        request.rootUUID = m_frameProxy->rootUUID();
        request.exclude = this->getUsrDataSpecialExcludes(); // 先获取特殊过滤项目，后面追加用户界面选择的
        for (auto &f : m_excludeFileItems) {
            if (f.fileType == File) {
                //过滤文件
                request.exclude.append(QString("/%1").arg(f.fileName));
            } else {
                //过滤文件夹
                if (f.children.isEmpty()) {
                    request.exclude.append(QString("/%1").arg(f.fileName));
                } else {
                    for (auto &c : f.children) {
                        if (c.fileType == File) {
                            request.exclude.append(QString("/%1/%2").arg(f.fileName).arg(c.fileName));
                        } else {
                            request.exclude.append(QString("/%1/%2").arg(f.fileName).arg(c.fileName));
                        }

                    }
                }
            }
        }

        m_excludes = request.exclude;
        m_userDataBackupSelectWidget->setTips(tr("Calculating file size..."));
        m_userDataBackupSelectWidget->setTipsStyleSheet("QLabel {"
                                                        "color: #000000;"
                                                        "}");
        m_userDataBackupSelectWidget->startSpinner();
        QJsonObject jsonObject = request.marshal();
        auto requestReply = m_recoveryInterface->CheckUserDataBackupSpace(Utils::JsonToQString(jsonObject));
        requestReply.waitForFinished();
        if (!requestReply.isValid()) {
            m_userDataBackupSelectWidget->stopSpinner();
            m_userDataBackupSelectWidget->setTips(tr(""));
            m_userDataBackupSelectWidget->setNextBtnEnabled(true);
        //    qCritical() << Q_FUNC_INFO << ", CheckUserDataBackupSpace failed, err: " << requestReply.error();
            onShowResult(false, CheckUserDataBackupSpace, tr("DBus error, please try again"));
            return;
        }
        if (requestReply.value() != OK) {
            m_userDataBackupSelectWidget->stopSpinner();
            m_userDataBackupSelectWidget->setTips(tr(""));
            m_userDataBackupSelectWidget->setNextBtnEnabled(true);
            onShowResult(false, OperateType::CheckUserDataBackupSpace);
            return;
        }
    }
}

void BackupModule::showUserDataBackupWidget(const QString &comboBoxText)
{
    if (m_userDataBackupWidget == nullptr) {
        m_userDataBackupWidget = new UserDataBackupWidget();
        connect(m_userDataBackupWidget, &UserDataBackupWidget::cancel, this, &BackupModule::onBack);
        connect(m_userDataBackupWidget, &UserDataBackupWidget::start, this,
                &BackupModule::onStartDataBackup);
    }

    QList<FileItem> tmpFileItems = m_fileItems;
    for (auto excludeItem : m_excludeFileItems) {
        for (auto &item : tmpFileItems) {
            if (item == excludeItem) {
                if (excludeItem.children.isEmpty()) {
                    // 过滤整个一级文件夹
                    tmpFileItems.removeAll(excludeItem);
                    continue;
                }

                // 过滤一级文件夹下面的部分文件夹或者文件
                for (auto childExcludeItem : excludeItem.children) {
                    item.children.removeAll(childExcludeItem);
                }
            }
        }
    }

    m_userDataBackupWidget->setFiles(tmpFileItems);

    auto partitionReply = m_recoveryInterface->ListPartition(true);
    partitionReply.waitForFinished();
    if (!partitionReply.isValid()) {
        qCritical() << partitionReply.error();
        onShowResult(false, UserDataBackup,
                     tr("DBus error, please try again"));
        return;
    }

    m_userDataBackupWidget->setComboBoxText(comboBoxText);
    m_userDataBackupWidget->setRemark(""); // clear
    m_frameProxy->setCurrentWidget(this, m_userDataBackupWidget);
}

void BackupModule::onReportCheckSpace(const QString &space)
{
    QJsonObject jsonObject = Utils::QStringToJson(space);
    int recoveryType = -1;
    if (jsonObject.contains("recoveryType")) {
        recoveryType = jsonObject.value("recoveryType").toInt(-1);
    }

    int errorCode = -1;
    if (jsonObject.contains("errCode")) {
        errorCode = jsonObject.value("errCode").toInt(-1);
    }

    int operateType = -1;
    if (jsonObject.contains("operateType")) {
        operateType = jsonObject.value("operateType").toInt(-1);
    }

    if (recoveryType == static_cast<int> (RecoveryType::Rsync) ||
        recoveryType == static_cast<int> (RecoveryType::OSTree)) {
        return this->systemBackup(operateType, errorCode);
    }
}

void BackupModule::systemBackup(int operateType, int errorCode)
{
    if (operateType == static_cast<int> (OperateType::CheckIncSystemBackupSpace)) {
        if (errorCode != OK) {
            m_frameProxy->setMenuDisabled(false);
            m_frameProxy->setWindowFuncClose(true);
            m_systemBackupWidget->setTips(
                    tr("Insufficient disk space. Please clean up first."));
            qCritical() << Q_FUNC_INFO << " systemBackup failed! errorCode = " << errorCode;
            return;
        }

        auto reply = m_recoveryInterface->SystemBackup(m_curSysBackupReq);
        reply.waitForFinished();
        if (!reply.isValid()) {
            onShowResult(false, SystemBackup,
                         tr("DBus error, please try again"));
            return;
        }

        int replayValue = reply.value();
        if (replayValue != ErrorCode::OK) {
            //    qInfo()<<"SystemBackup replayValue = "<<replayValue;
            // TODO: 可以根据具体的错误码显示更详细的错误提示信息，先简单显示
            onShowResult(false, SystemBackup,
                         tr("Sorry, backup failed!"));
            return;
        }

        onShowProgress(tr("Backing up..."),
                       tr("Time remaining: "),
                       tr("To avoid data loss, please do not use your computer during the process."));
        return;
    }
}
