/*****************************************************************************
 *
 *  Copyright (C) 2003 Cdric Brgardis <cedric.bregardis@free.fr>
 *
 *  This file is part of BRIQUOLO
 *
 *  BRIQUOLO is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  BRIQUOLO is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with BRIQUOLO; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *****************************************************************************/
#ifndef _ELEMENTCOLLISION
#define _ELEMENTCOLLISION

#include <map>
#include <string>
#include "ElementGraphique.h"
#include "Generateur.h"

using namespace std;

class Balle;
class Struct_Collision;

class Point
{
  public:
    float x, y;
    Point(float p_X, float p_Y):x(p_X), y(p_Y)
    {
    }
};

class ElementCollision: public ElementGraphique
{
  protected:
    // **** Types locaux  la classe ****
    typedef map<string, string> Map_Params;
    typedef Map_Params::iterator It_Map_Params;
    // **********************************

  protected:
    static ElementCollision * _DernierElementCollision;
    static Balle * _DerniereBalle;
    static float _DernierTempsCollision;


    unsigned int _Niveau;
    string _LigneOption;
    Map_Params _Params;

  protected:
    float _Determinant(float d1x, float d1y, float d2x, float d2y);
    float _Modulo(float a, float b);

    Struct_Collision _CircleCollisionWithoutRot(float c1ix, float c1iy, float r1, float v1x, float v1y,
                                                float brick_center_x, float brick_center_y,
                                                float c2ix, float c2iy, float r2, float v2x, float v2y,
                                                float tmax);
    Struct_Collision _CircleCollisionWithRotSearch(float c1ix, float c1iy, float r1, float v1x, float v1y,
                                                   float brick_center_x, float brick_center_y, float brick_rot_i,
                                                   float teta0, float ra, float r2, float v2x, float v2y, float v2r,
                                                   float tmin, float dtmin,
                                                   float tmax, float dtmax);

    Struct_Collision _CircleCollisionWithRot(float c1ix, float c1iy, float r1, float v1x, float v1y,
                                             float brick_center_x, float brick_center_y, float brick_rot_i,
                                             float c2ix, float c2iy, float r2, float v2x, float v2y, float v2r,
                                             float tmax);
    Struct_Collision _CircleCollision(float c1ix, float c1iy, float r1, float v1x, float v1y,
                                      float brick_center_x, float brick_center_y, float brick_rot_i,
                                      float c2ix, float c2iy, float r2, float v2x, float v2y, float v2r,
                                      float tmax);

    Struct_Collision _SegmentCollision(float c1ix, float c1iy, float r1, float v1x, float v1y,
                                       float brick_center_x, float brick_center_y, float brick_rot_i,
                                       float teta0, float ra, float gama0, float rb,
                                       float vtx, float vty, float vr,
                                       float tmin, float dtmin, float detmin,
                                       float tmax, float dtmax, float detmax);

    Struct_Collision _SegmentCollision(float c1ix, float c1iy, float r1, float v1x, float v1y,
                                       float brick_center_x, float brick_center_y, float brick_rot_i,
                                       float ax, float ay, float bx, float by, float vtx, float vty, float vr,
                                       float tmin, float tmax);

    void _Rebond(float nx, float ny, float vix, float viy, float & vfx, float & vfy);

    virtual void _AnalyserLigne();

  public:
    ElementCollision();
    ElementCollision(const ElementCollision & p_ElementCollision);
    virtual ~ElementCollision();


    /**
       Cette mthode statique permet de mmoriser le dernier lment responsable d'une collision avec une balle.
     */
    static void SetDerniereCollision(ElementCollision * p_Element, Balle * p_Balle);
    
    /**
       Permet de dterminer si la balle passe en paramtre entre en collision avec cet ElementCollision.
       Cette mthode doit tre implmente par la classe spcialisant cette classe abstraite.
       @return La structure de collision. Elle indique (entre autre) le moment (TempsCollision) de la collision si elle a lieu.
               En comparant les diffrents TempsCollision des diffrentes structures renvoy par les diffrents lment, on
               peut savoir quel est l'lment sur lequel la collision a lieu en premier.
     */
    virtual Struct_Collision TesterCollision(Balle * p_Balle) = 0;

    /**
       Aprs avoir dtermin l'lment entrant en collision avec la balle (grace  TesterCollision) cette mthode est appel sur
       l'lment en question. 
       @return <i>true</i> si l'lment disparait, <i>false</i> sinon.
     */
    virtual bool Toucher(Balle * p_Balle, double p_Temps);

    /**
       Permet de dplacer l'lement jusqu'au temps pass en paramtre. Cette mthode est appele aprs aprs que le moment de la
       la collision ait t dtermin  l'aide de TesterCollision.
     */
    virtual void Deplacer(double p_Temps); 

    /**
       Permet de crer un gnrateur de particules correspondant  une collision. Elle est appele lorsque l'lement 
       est responsable d'une collision. Si aucun gnrateur de particule ne doit tre cr, alors la mthode renvoie NULL.
     */
    virtual Generateur * CreerGenerateurParticules();

    /**
       Permet de rcuprer la ligne d'option associ  l'lment.
       @return La chaine de caractre reprsentant l'lment.
     */
    string GetLigneOption() const;

    /**
       Permet de dfinir la ligne d'option associ  l'lment.
       @param p_Ligne : la chaine de caractre reprsentant l'lment.
     */
    void SetLigneOption(const string & p_Ligne);

    virtual Point GetVisualCenter();


    /** 
     * Indique si la dernire collision peut tre considre comme achev.
     * Lorsqu'une collision se produit, il est ncssaire d'attendre
     * que la balle s'carte lgrement avant de reconsidrer une nouvelle
     * collision sur cet lment.
     * 
     * @return <i>true</i> si on peut tester une nouvelle collision, <i>false</i> sinon.
     */
    bool IsDerniereCollisionOk(Balle * p_Balle);
};


struct Struct_Collision
{
    double VitesseX, VitesseY;
    double PositionX, PositionY;
    //double PositionCollision;
    double TempsCollision;
    double AddFacteurCollision;
    ElementCollision * ResponsableCollision;

    Struct_Collision()
    : VitesseX(0.0), VitesseY(0.0), 
      PositionX(0.0), PositionY(0.0),
      TempsCollision(0.0),
      AddFacteurCollision(1), 
      ResponsableCollision(NULL)
    {
    }
};


#endif
