/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "JSWebAssemblyException.h"

#if ENABLE(WEBASSEMBLY)

#include "AuxiliaryBarrierInlines.h"
#include "JSBigInt.h"
#include "JSCJSValueInlines.h"
#include "JSCellInlines.h"
#include "JSWebAssemblyHelpers.h"
#include "StructureInlines.h"

namespace JSC {

const ClassInfo JSWebAssemblyException::s_info = { "WebAssembly.Exception", &Base::s_info, nullptr, nullptr, CREATE_METHOD_TABLE(JSWebAssemblyException) };

JSWebAssemblyException::JSWebAssemblyException(VM& vm, Structure* structure, const Wasm::Tag& tag, FixedVector<uint64_t>&& payload)
    : Base(vm, structure)
    , m_tag(Ref { tag })
    , m_payload(WTFMove(payload))
{
}

void JSWebAssemblyException::finishCreation(VM& vm)
{
    Base::finishCreation(vm);
    ASSERT(inherits(vm, info()));
}

template<typename Visitor>
void JSWebAssemblyException::visitChildrenImpl(JSCell* cell, Visitor& visitor)
{
    Base::visitChildren(cell, visitor);

    auto* exception = jsCast<JSWebAssemblyException*>(cell);
    const Wasm::Signature& signature = exception->tag().signature();
    for (unsigned i = 0; i < signature.argumentCount(); ++i) {
        if (isRefType(signature.argument(i)))
            visitor.append(bitwise_cast<WriteBarrier<Unknown>>(exception->payload()[i]));
    }
}

DEFINE_VISIT_CHILDREN(JSWebAssemblyException);

void JSWebAssemblyException::destroy(JSCell* cell)
{
    static_cast<JSWebAssemblyException*>(cell)->JSWebAssemblyException::~JSWebAssemblyException();
}

JSValue JSWebAssemblyException::getArg(JSGlobalObject* globalObject, unsigned i) const
{
    const Wasm::Signature& signature = tag().signature();
    ASSERT(i < signature.argumentCount());
    return toJSValue(globalObject, signature.argument(i), payload()[i]);
}

} // namespace JSC

#endif // ENABLE(WEBASSEMBLY)
