/*
   Copyright (C) 2017-2018 by the Battle for Wesnoth Project https://www.wesnoth.org/

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.

   See the COPYING file for more details.
*/

#pragma once

#include "scripting/lua_common.hpp"

#include <type_traits>
#include <boost/mpl/not.hpp>
#include <boost/mpl/and.hpp>
#include <boost/mpl/has_xxx.hpp>
#include "tstring.hpp"
#include "map/location.hpp"
#include "lua/lauxlib.h"
#include "lua/lua.h"

#include "utils/type_trait_aliases.hpp"

#include <cassert>

class enum_tag;

namespace lua_check_impl
{
	namespace detail
	{
		BOOST_MPL_HAS_XXX_TRAIT_DEF(value_type)
		BOOST_MPL_HAS_XXX_TRAIT_DEF(iterator)
		BOOST_MPL_HAS_XXX_TRAIT_DEF(size_type)
		BOOST_MPL_HAS_XXX_TRAIT_DEF(reference)
		BOOST_MPL_HAS_XXX_TRAIT_DEF(key_type)
		BOOST_MPL_HAS_XXX_TRAIT_DEF(mapped_type)
		BOOST_MPL_HAS_XXX_TRAIT_DEF(first_type)
		BOOST_MPL_HAS_XXX_TRAIT_DEF(second_type)
	}

	template<typename T, typename T2 = utils::remove_reference_t<T>>
	struct is_container
		: boost::mpl::bool_<
			detail::has_value_type<T2>::value &&
			detail::has_iterator<T2>::value &&
			detail::has_size_type<T2>::value &&
			detail::has_reference<T2>::value
		>
	{};

	template<typename T, typename T2 = utils::remove_reference_t<T>>
	struct is_map
		: boost::mpl::bool_<
			detail::has_key_type<T2>::value &&
			detail::has_mapped_type<T2>::value
		>
	{};

	template<typename T, typename T2 = utils::remove_reference_t<T>>
	struct is_pair
		: boost::mpl::bool_<
			detail::has_first_type<T2>::value &&
			detail::has_second_type<T2>::value
		>
	{};

	template<typename T>
	using remove_constref = utils::remove_const_t<utils::remove_reference_t<utils::remove_const_t<T>>>;

	//std::string
	template<typename T>
	utils::enable_if_t<std::is_same<T, std::string>::value, std::string>
	lua_check(lua_State *L, int n)
	{
		return luaL_checkstring(L, n);
	}
	template<typename T>
	utils::enable_if_t<std::is_same<T, std::string>::value, void>
	lua_push(lua_State *L, const T& val)
	{
		lua_pushlstring(L, val.c_str(), val.size());
	}

	//config
	template<typename T>
	utils::enable_if_t<std::is_same<T, config>::value, config>
	lua_check(lua_State *L, int n)
	{
		return luaW_checkconfig(L, n);
	}
	template<typename T>
	utils::enable_if_t<std::is_same<T, config>::value, void>
	lua_push(lua_State *L, const config& val)
	{
		luaW_pushconfig(L, val);
	}

	//location
	template<typename T>
	utils::enable_if_t<std::is_same<T, map_location>::value, map_location>
	lua_check(lua_State *L, int n)
	{
		return luaW_checklocation(L, n);
	}
	template<typename T>
	utils::enable_if_t<std::is_same<T, map_location>::value, void>
	lua_push(lua_State *L, const map_location& val)
	{
		luaW_pushlocation(L, val);
	}

	//enums generated by MAKE_ENUM
	template<typename T>
	utils::enable_if_t<std::is_base_of<enum_tag, T>::value, T>
	lua_check(lua_State *L, int n)
	{
		T val;
		std::string str = lua_check_impl::lua_check<std::string>(L, n);
		if(!val.parse(str))
		{
			luaL_argerror(L, n, ("cannot convert " + str + " to enum " + T::name()).c_str());
		}
		return val;
	}
	template<typename T>
	utils::enable_if_t<std::is_base_of<enum_tag, T>::value, void>
	lua_push(lua_State *L, T val)
	{
		lua_check_impl::lua_push(L, val.to_string());
	}

	//t_string
	template<typename T>
	utils::enable_if_t<std::is_same<T, t_string>::value, t_string>
	lua_check(lua_State *L, int n)
	{
		return luaW_checktstring(L, n);
	}
	template<typename T>
	utils::enable_if_t<std::is_same<T, t_string>::value, void>
	lua_push(lua_State *L, const t_string& val)
	{
		luaW_pushtstring(L, val);
	}

	//bool
	template<typename T>
	utils::enable_if_t<std::is_same<T, bool>::value, bool>
	lua_check(lua_State *L, int n)
	{
		return luaW_toboolean(L, n);
	}
	template<typename T>
	utils::enable_if_t<std::is_same<T, bool>::value, void>
	lua_push(lua_State *L, bool val)
	{
		lua_pushboolean(L, val);
	}

	//double, float
	template<typename T>
	utils::enable_if_t<std::is_floating_point<T>::value, T>
	lua_check(lua_State *L, int n)
	{
		return luaL_checknumber(L, n);
	}
	template<typename T>
	utils::enable_if_t<std::is_floating_point<T>::value, void>
	lua_push(lua_State *L, T val)
	{
		lua_pushnumber(L, val);
	}

	//integer types
	template<typename T>
	utils::enable_if_t<std::is_integral<T>::value && !std::is_same<T, bool>::value, T>
	lua_check(lua_State *L, int n)
	{
		return luaL_checkinteger(L, n);
	}

	template<typename T>
	utils::enable_if_t<std::is_integral<T>::value && !std::is_same<T, bool>::value, void>
	lua_push(lua_State *L, T val)
	{
		lua_pushnumber(L, val);
	}

	//std::pair
	//Not sure if the not_<is_const> is required; only (maybe) if std::map matches is_container
	template<typename T>
	utils::enable_if_t<is_pair<T>::value && !std::is_const<typename T::first_type>::value, T>
	lua_check(lua_State *L, int n)
	{
		T result;
		if (lua_istable(L, n)) {
			lua_rawgeti(L, n, 1);
			result.first = lua_check<const typename T::first_type&>(L, -1);
			lua_rawgeti(L, n, 2);
			result.second = lua_check<const typename T::second_type&>(L, -1);
			lua_pop(L, 2);
		}
		return result;
	}
	template<typename T>
	utils::enable_if_t<is_pair<T>::value && !std::is_const<typename T::first_type>::value, void>
	lua_push(lua_State *L, const T& val)
	{
		lua_newtable(L);
		lua_push<const typename T::first_type&>(L, val.first);
		lua_rawseti(L, -2, 1);
		lua_push<const typename T::second_type&>(L, val.second);
		lua_rawseti(L, -2, 2);
	}

	//std::vector and similar but not std::string
	template<typename T>
	utils::enable_if_t<is_container<T>::value && !std::is_same<T, std::string>::value, T>
	lua_check(lua_State * L, int n)
	{
		if (lua_istable(L, n))
		{
			T res;
			for (int i = 1, i_end = lua_rawlen(L, n); i <= i_end; ++i)
			{
				lua_rawgeti(L, n, i);
				res.push_back(lua_check_impl::lua_check<remove_constref<typename T::reference>>(L, -1));
				lua_pop(L, 1);
			}
			return res;
		}
		else
		{
			luaL_argerror(L, n, "Table expected");
			throw "luaL_argerror returned"; //shouldn't happen, luaL_argerror always throws.
		}
	}

#if defined(__GNUC__) && !defined(__clang__)
#if __GNUC__ < 4 || (__GNUC__ == 4 && __GNUC_MINOR__ < 8 )
// 'list.size()' below is unsigned for some (most but not all) list types.
#pragma GCC diagnostic ignored "-Wtype-limits"
#endif
#endif

	//also accepts things like std::vector<int>() | std::adaptors::transformed(..)
	template<typename T>
	utils::enable_if_t<
		is_container<T>::value && !std::is_same<T, std::string>::value && !is_map<T>::value
		, void
	>
	lua_push(lua_State * L, const T& list )
	{
		// NOTE: T might be some boost::iterator_range type where size might be < 0. (unfortunately in this case size() does not return T::size_type)
		assert(list.size() >= 0);
		lua_createtable(L, list.size(), 0);
		int i = 1;
		for(typename T::const_iterator iter = list.begin(); iter != list.end(); ++iter) {
			lua_check_impl::lua_push<remove_constref<typename T::reference>>(L, *iter);
			lua_rawseti(L, -2, i++);
		}
	}


	//accepts std::map TODO: add a check function for that
	template<typename T>
	utils::enable_if_t<is_map<T>::value, void>
	lua_push(lua_State * L, const T& map )
	{
		lua_newtable(L);
		for(const typename T::value_type& pair : map)
		{
			lua_check_impl::lua_push<remove_constref<typename T::key_type>>(L, pair.first);
			lua_check_impl::lua_push<remove_constref<typename T::mapped_type>>(L, pair.second);
			lua_settable(L, -3);
		}
	}

}

template<typename T>
lua_check_impl::remove_constref<T> lua_check(lua_State *L, int n)
{
	//remove possible const& to make life easier for the impl namespace.
	return lua_check_impl::lua_check<lua_check_impl::remove_constref<T>>(L, n);
}

template<typename T>
void lua_push(lua_State *L, const T& val)
{
	return lua_check_impl::lua_push<lua_check_impl::remove_constref<T>>(L, val);
}
